import os
import ssl
from typing import Optional, Dict, List

import requests
import json
from enum import Enum
import logging
from datetime import datetime, timedelta

from dataclasses import dataclass

logging.basicConfig(format="%(levelname)s:%(message)s", level=logging.INFO)
ssl._create_default_https_context = ssl._create_unverified_context


def calculate_acceptable_time_ranges(testing_data):
    now = datetime.now()

    if testing_data['night_run'] and (10 < now.hour < 22):
        start = datetime(now.year, now.month, now.day, 22, 0, 0)
    else:
        start = now + timedelta(minutes=15)

    finish = start + timedelta(hours=testing_data['MAX_DURATION'])

    return [{
        "startTs": start.timestamp() * 1000,
        "endTs": finish.timestamp() * 1000
    }]


@dataclass(init=True)
class Settings:
    TOKEN: str
    QUOTA_CODE: str
    MAX_QUOTA: Optional[float]  # (may be change) max acceptable quota (in working hours) or None if does not matter
    MAX_DURATION: Optional[float]  # (may be change) max acceptable duration of task in hours

    acceptable_time_ranges: list
    booking: dict

    BASE_URL: str = "https://booking.yandex-team.ru/"
    PATH_TO_CERTIFICATE: str = os.path.join(os.path.dirname(__file__),
                                            'YandexInternalRootCA.crt')  # Path to certificate if executed locally
    PRICE_PER_WH: float = 120.0  # do not change
    DURATION_MAX_RATIO: float = 1.2  # max ratio between actual duration and MAX_DURATION. 1.2 means 120% of MAX_DURATION is ok but no more

    # todo: choose start dates here
    # time ranges in which you are ok to start booking


GeneralSetting = Settings(
    TOKEN='xxx',
    QUOTA_CODE="qs_testpalm_separate_mobmail_android_ru",
    MAX_QUOTA=5, MAX_DURATION=10,
    acceptable_time_ranges=[],
    booking={}
)


class Speed(Enum):
    SLOW = 0  # slow is for bookings with low priority
    NORMAL = 1  # for everyday bookings
    URGENT = 2  # for extremely fast bookings with short deadline (higher costs)

    def lower(self):
        if self == Speed.URGENT:
            return Speed.NORMAL
        elif self == Speed.NORMAL:
            return Speed.SLOW
        else:
            raise Exception("The speed mode is already the slowest")

    def higher(self):
        if self == Speed.SLOW:
            return Speed.NORMAL
        elif self == Speed.NORMAL:
            return Speed.URGENT
        else:
            raise Exception("The speed mode is already the highest possible.")


def get_next_range(start_ts: int):
    global GeneralSetting
    for item in GeneralSetting.acceptable_time_ranges:
        if item["startTs"] > start_ts:
            return item
    return None


def get_time_range(start_ts: int):
    global GeneralSetting
    for item in GeneralSetting.acceptable_time_ranges:
        if item["startTs"] == start_ts:
            return item


def sort_time_ranges(time_ranges: List):
    return sorted(time_ranges, key=lambda k: k["startTs"], reverse=True)


def validate_time_ranges(time_ranges: List):
    for time_range in time_ranges:
        if time_range["startTs"] >= time_range["endTs"]:
            raise Exception(
                "Time range has endTs "
                + str(time_range["endTs"])
                + " is less than startTs "
                + str(time_range["startTs"])
            )
    if len(time_ranges) == 0:
        raise Exception("Time ranges cannot be empty")


def create_booking(title: str, params: Dict, estimation: Dict):
    global GeneralSetting

    url = GeneralSetting.BASE_URL + "api/bookings/assessor-testing/default"
    booking_data = {"title": title,
                    "params": params,
                    "estimate": estimation,
                    "metaData": {
                        "subscribers": [],
                        "customData": {
                            "allowStartBeforeBookingStartTs": 'true'
                        }
                    }
                    }
    logging.info("Attempt to create: %s", booking_data)
    response = requests.post(
        url,
        data=json.dumps(booking_data),
        headers={
            "Authorization": "OAuth " + GeneralSetting.TOKEN,
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
        verify=GeneralSetting.PATH_TO_CERTIFICATE,
        timeout=40,
    )
    if response.status_code == 200:
        logging.info("Created booking: %s", response.json())
        GeneralSetting.booking = response.json()
        return response.json()
    else:
        logging.error(response.status_code, response.json())
        return None


def check_duration(body: Dict, estimate_result: Dict):
    global GeneralSetting
    logging.info("Checking duration...")
    current_duration = (
            (estimate_result["deadlineTs"] - estimate_result["startTs"])
            / 1000.0
            / 60.0
            / 60.0
    )

    if (
            GeneralSetting.MAX_DURATION is not None
            and GeneralSetting.MAX_DURATION != 0
            and current_duration / GeneralSetting.MAX_DURATION > GeneralSetting.DURATION_MAX_RATIO
    ):
        logging.info(
            "Duration is %s with MAX_DURATION %s", current_duration, GeneralSetting.MAX_DURATION
        )
        GeneralSetting.MAX_DURATION = current_duration + 4
        # current_speed = Speed[body["speedMode"]].higher()
        # logging.info(
        #     "Speed mode was changed from %s to %s",
        #     body["speedMode"],
        #     current_speed.name,
        # )
        # body["speedMode"] = current_speed.name
        return False, body
    else:
        logging.info("Duration is ok!")
        return True, body


def check_max_quota(body: Dict, estimate_result: Dict):
    global GeneralSetting

    logging.info("Checking max quota...")
    if GeneralSetting.MAX_QUOTA is not None and GeneralSetting.PRICE_PER_WH * GeneralSetting.MAX_QUOTA < \
            estimate_result["minQuota"]:
        diff_quota = GeneralSetting.MAX_QUOTA * GeneralSetting.PRICE_PER_WH - estimate_result["minQuota"]
        logging.info("Quota is more than %s", abs(diff_quota))
        if body["speedMode"] == Speed.SLOW.name:
            GeneralSetting.MAX_QUOTA = int(estimate_result['resourceVolume']) + 1
            GeneralSetting.MAX_DURATION = GeneralSetting.MAX_QUOTA * 2
        else:
            current_speed = Speed[body["speedMode"]].lower()
            logging.info(
                "Speed mode was changed from %s to %s",
                body["speedMode"],
                current_speed.name,
            )
            body["speedMode"] = current_speed.name
        return False, body
    else:
        logging.info("Max quota is ok!")
        return True, body


def check_time_range(body: Dict, estimate_result: Dict):
    global GeneralSetting

    logging.info("Checking start time...")
    time_range = get_time_range(body["startTsFrom"])
    if (
            time_range["startTs"] < estimate_result["startTs"]
            or estimate_result["startTs"] > time_range["endTs"]
    ):
        logging.info("StartsTs was moved to %s", estimate_result["startTs"])
        GeneralSetting.acceptable_time_ranges.append({"startTs": estimate_result['startTs'],
                                                      "endTs": estimate_result['deadlineTs']})
        next_time_range = get_next_range(body["startTsFrom"])
        if next_time_range is None:
            raise Exception("No acceptable time periods!")
        body["startTsFrom"] = next_time_range["startTs"]
        logging.info("Changed startTs is %s", body["startTsFrom"])
        return False, body
    else:
        logging.info("StartTs is ok!")
        return True, body


def estimate(title: str, body: Dict):
    """
    tries to estimate time and checks with parameters given
    checks with following logic:
    1. first tries to booking in the beginning of the earliest time period.
    2. if start time is not in the earliest time range, tries to booking to the next time range or throws Exception
    3. if start time is ok, it checks task duration. If MAX_DURATION is set and duration is more than max,
    the speed mode is increased. If speed mode is already the highest, then exception is raised.
    4. If duration is ok, it checks max quota. If MAX_QUOTA is set and is more than max, it tries to set lower speed.
    Id the speed mode is the lowest, then it raises exception.
    5. If all parameters are ok, then the booking is created.
    :param title: name of the booking
    :param body: parameters to estimate according to type
    :return: booking or None
    """
    global GeneralSetting
    logging.info("Estimating ", body)
    url = GeneralSetting.BASE_URL + "api/bookings/assessor-testing/default/estimate"
    response = requests.post(
        url,
        data=json.dumps(body),
        headers={
            "Authorization": "OAuth " + GeneralSetting.TOKEN,
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
        verify=GeneralSetting.PATH_TO_CERTIFICATE,
        timeout=40,
    )
    if response.status_code != 200:
        logging.info(response.status_code)
        logging.info(response.json())
        return None

    estimate_result = response.json()
    logging.info("Estimated result %s", estimate_result)

    is_time_ok, body = check_time_range(body, estimate_result)

    if not is_time_ok:
        estimate(title, body)
    else:
        is_duration_ok, estimate_request = check_duration(body, estimate_result)
        if not is_duration_ok:
            estimate(title, estimate_request)
        else:
            is_quota_ok, estimate_request = check_max_quota(body, estimate_result)
            if not is_quota_ok:
                estimate(title, estimate_request)
            else:
                return create_booking(title, body, estimate_result)


def book_custom_volume(
        title: str,
        volume_amount: float,
        environment_distribution: Dict,
        speed: Enum,
        start_ts: int,
):
    """

    :param title: name of the booking
    :param volume_amount: additional volume for the booking in БО
    :param environment_distribution: distribution of environments that would be used with rate.
        Summary of rate should be equal to 1. Example,  {"64": 0.2, "63": 0.8}.
        Key is code for environment and value is for rate.
        More on https://booking.yandex-team.ru/#/help?section=api-envs
    :param speed: type of speed mode. NORMAL is recommended
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    global GeneralSetting

    body = {
        "quotaSource": GeneralSetting.QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [],
            "customVolume": {
                "amount": volume_amount,
                "environmentDistribution": environment_distribution,
            },
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def book_st_filter(
        title: str, st_filter: str, speed: Enum, custom_volume: float, start_ts: int
):
    """

    :param title: name of the booking
    :param st_filter: link to Startrek filter.Example, "https://st.yandex-team.ru/issues/29390"
    :param speed: type of speed mode. NORMAL is recommended
    :param custom_volume: additional volume for the booking in БО
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    global GeneralSetting

    body = {
        "quotaSource": GeneralSetting.QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [{"type": "ST_TICKET_FILTER", "values": [st_filter]}],
            "customVolume": custom_volume if custom_volume > 0 else None,
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def book_st_tickets(
        title: str, st_tickets: List, speed: Enum, custom_volume: float, start_ts: int
):
    """

    :param title: name of the booking
    :param st_tickets: array of links to Startek tickets. Example, ["https://st.yandex-team.ru/TESTING-1", "https://st.yandex-team.ru/TESTING-2"]
    :param speed: type of speed mode. NORMAL is recommended
    :param custom_volume: additional volume for the booking in БО
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None

    book_st_tickets(
            f"Testing Resolve {key}", [f"https://st.yandex-team.ru/{key}"], starting_speed, 0, startTs,
        )

    """
    global GeneralSetting

    body = {
        "quotaSource": GeneralSetting.QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [{"type": "ST_TICKET", "values": st_tickets}],
            "customVolume": custom_volume if custom_volume > 0 else None,
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def book_testpalm_version(
        title: str, versions: List, speed: Enum, custom_volume: float, start_ts: int
):
    """

    :param title: name of the booking
    :param versions:  array of links to Testpalm versions. Example, ["https://testpalm.yandex-team.ru/test/version/v_1", "https://testpalm.yandex-team.ru/test/version/v_2"]
    :param speed: type of speed mode. NORMAL is recommended
    :param custom_volume: additional volume for the booking in БО
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    global GeneralSetting

    body = {
        "quotaSource": GeneralSetting.QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [{"type": "TEST_PALM", "values": versions}],
            "customVolume": custom_volume if custom_volume > 0 else None,
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def get_start_time():
    """
    Sorts and validates time ranges
    If all valid, returns startTs of first time range
    :return:
    """
    global GeneralSetting
    GeneralSetting.acceptable_time_ranges = sort_time_ranges(GeneralSetting.acceptable_time_ranges)
    validate_time_ranges(GeneralSetting.acceptable_time_ranges)
    return GeneralSetting.acceptable_time_ranges[0]["startTs"]


def booking_version_testpalm(project, version, starting_speed=Speed.NORMAL):
    global GeneralSetting
    testing_data = json.load(open(os.path.join(os.path.dirname(__file__), f'testing_data_{project}.json'), encoding='utf-8'))

    GeneralSetting = Settings(
        TOKEN=testing_data['AUTH_ST'],
        QUOTA_CODE=testing_data['bron']['QUOTA_CODE'],
        MAX_QUOTA=testing_data['bron']['MAX_QUOTA'],
        MAX_DURATION=testing_data['bron']['MAX_DURATION'],
        booking={},
        acceptable_time_ranges=calculate_acceptable_time_ranges(testing_data['bron'])
    )

    # First you need to get startTs for booking
    startTs = get_start_time()

    try:
        book_testpalm_version(
            f"Version - {testing_data['version_app']}, Suites: {testing_data['scenario']}, Part - {testing_data['part']}",
            [f"{testing_data['testpalm']}/{version}"],
            starting_speed,
            0,
            startTs,
        )
        return GeneralSetting.booking
    except Exception as e:
        print(e)
