from dataclasses import dataclass
from typing import List, Optional, Dict
from furl import furl
import requests
import json
import re
from requests import Response
from requests.adapters import HTTPAdapter
from urllib3 import Retry
import logging
import os
import statface_client


OAUTH_APPMETRICA = os.environ['OAUTH_TOKEN_APPMETRICA']
CONFIG = {
    '40010': {
        'start_date': '2019-10-21',
        'end_date': '2019-11-03'
    },
    '40070': {
        'start_date': '2020-01-20',
        'end_date': '2020-02-03'
    }
}

logger = logging.getLogger("statface_client")
logging.basicConfig(level=logging.WARNING)
stat_client = statface_client.ProductionStatfaceClient()


@dataclass
class Extension:
    nce = 'ru.yandex.mail.notificationcontentextension'
    nse = 'ru.yandex.mail.notificationserviceextension'


class AppMetricaClient:
    def __init__(self, os, auth, api_key, host='https://api.appmetrica.yandex.ru'):
        self.__auth = auth
        self.__os = os
        self.__api_key = api_key
        self.__host = host
        self.__headers = {
            'Authorization': f'OAuth {self.__auth}',
            'Content-Type': 'application/json'
        }

    @staticmethod
    def __connect():
        r = requests.Session()
        retries = Retry(total=5,
                        backoff_factor=0.1,
                        status_forcelist=[500, 502, 503, 504],
                        method_whitelist=frozenset(['GET', 'POST']))
        r.mount('https://', HTTPAdapter(max_retries=retries))
        return r

    def __make_get_request(self, url: str) -> Response:
        response = self.__connect().get(url, headers=self.__headers, verify=False)
        response.raise_for_status()
        return response

    def get_crashes_count_and_device_percent_by_period(self, start_date: str, end_date: str, app_version: Optional[str] = None, extension: str = None):
        os = self.__os.lower()

        def format_os(os: str) -> str:
            if os == 'ios':
                return 'iOS'
            elif os == 'android':
                return 'Android'
            else:
                raise ValueError('Unknown OS')

        app_version_filter, extension_filter = '', ''
        if app_version:
            app_version_filter = f" AND appVersionDetails == '{app_version} ({format_os(os)})'"

        if extension:
            extension_filter = f" AND appID == '{extension}'"

        url = furl(self.__host) \
            .add(path=['stat', 'v1', 'data', 'bytime']) \
            .add(args={
                'filters': f"ym:cr2:operatingSystemInfo=='{os}'" + app_version_filter + extension_filter,
                'id': self.__api_key,
                'date1': start_date,
                'date2': end_date,
                'metrics': ','.join([
                    'ym:cr2:crashes',
                    'ym:cr2:crashDevices',
                    'ym:cr2:crashesDevicesPercentage'
                ]),
                'sort': '-ym:cr2:crashes',
                'debug': 'json',
                'accuracy': 1,
                'proposedAccuracy': True,
                'group': 'day'
            }
        )

        response = self.__make_get_request(url)
        return response.json()

    def get_crashes_count_by_period(self, start_date: str, end_date: str, app_version: Optional[str] = None, extension: str = None):
        os = self.__os.lower()

        def format_os(os: str) -> str:
            if os == 'ios':
                return 'iOS'
            elif os == 'android':
                return 'Android'
            else:
                raise ValueError('Unknown OS')

        app_version_filter, extension_filter = '', ''
        if app_version:
            app_version_filter = f" AND appVersionDetails == '{app_version} ({format_os(os)})'"

        if extension:
            extension_filter = f" AND appID == '{extension}'"

        url = furl(self.__host) \
            .add(path=['stat', 'v1', 'data', 'bytime']) \
            .add(args={
                'filters': f"ym:cr2:operatingSystemInfo=='{os}'" + app_version_filter + extension_filter,
                'id': self.__api_key,
                'date1': start_date,
                'date2': end_date,
                'metrics': ','.join([
                    'ym:cr2:crashes',
                    'ym:cr2:crashDevices'
                ]),
                'sort': '-ym:cr2:crashes',
                'debug': 'json',
                'accuracy': 1,
                'proposedAccuracy': True,
                'group': 'day'
            }
        )

        response = self.__make_get_request(url)
        return response.json()

    def get_crashes_free_session_and_devices(self, start_date: str, end_date: str, app_version: str = None, extension: str = None):
        os = self.__os.lower()

        def format_os(os: str) -> str:
            if os == 'ios':
                return 'iOS'
            elif os == 'android':
                return 'Android'
            else:
                raise ValueError('Unknown OS')

        app_version_filter, extension_filter = '', ''
        if app_version:
            app_version_filter = f" AND appVersionDetails == '{app_version} ({format_os(os)})'"

        if extension:
            extension_filter = f" AND appID == '{extension}'"

        url = furl(self.__host) \
            .add(path=['stat', 'v1', 'data', 'bytime']) \
            .add(args={
                'filters': f"ym:cr2:operatingSystemInfo=='{os}'" + app_version_filter + extension_filter,
                'id': self.__api_key,
                'date1': start_date,
                'date2': end_date,
                'metrics': ','.join([
                    'ym:cr2:crashesFreeSessionsPercentage',
                    'ym:cr2:crashesFreeDevicesPercentage'
                ]),
                'sort': '-ym:cr2:crashesFreeSessionsPercentage',
                'debug': 'json',
                'accuracy': 1,
                'proposedAccuracy': True,
                'group': 'day'
            }
        )

        response = self.__make_get_request(url)
        return response.json()

    def get_crash_stat(self, start_date: str, end_date: str, limit: int, app_version: str = None, extension: str = None) -> List:
        os = self.__os.lower()

        def format_os(os: str) -> str:
            if os == 'ios':
                return 'iOS'
            elif os == 'android':
                return 'Android'
            else:
                raise ValueError('Unknown OS')

        app_version_filter, extension_filter = '', ''
        if app_version:
            app_version_filter = f" AND appVersionDetails == '{app_version} ({format_os(os)})'"

        if extension:
            extension_filter = f" AND appID == '{extension}'"

        url = furl(self.__host) \
            .add(path=['stat', 'v1', 'data']) \
            .add(args={
                'filters': f"ym:cr2:operatingSystemInfo=='{os}'" + app_version_filter + extension_filter,
                'id': self.__api_key,
                'date1': start_date,
                'date2': end_date,
                'metrics': ','.join([
                    'ym:cr2:crashes',
                    'norm(ym:cr2:crashes)',
                    'ym:cr2:crashDevices',
                    'norm(ym:cr2:crashDevices)',
                    'ym:cr2:crashesDevicesPercentage',
                    'ym:cr2:crashesUndecoded',
                ]),
                'dimensions': 'ym:cr2:crashGroupObj',
                'sort': '-ym:cr2:crashes',
                'offset': 1,
                'limit': limit,
                'debug': 'json',
                'accuracy': 1,
                'proposedAccuracy': True
            }
        )

        response = self.__make_get_request(url)
        return response.json()


def open_json(file_name):
    with open(file_name) as f:
        d = json.load(f)
        return d


def save_data_to_json_file(file_name: str, data) -> None:
    with open(file_name, 'w', encoding='utf-8') as f:
        json.dump(data, f, ensure_ascii=False, indent=4)


def format_crash_data(data, version, extension, save_to_file: bool = None):
    data = data['data']
    print(data)
    result = list(filter(lambda x:
                        x is not None,
                        map(lambda x: {
                            'title': x['dimensions'][0]['name'] + ' ' + x['dimensions'][0]['method_name'],
                            'id': x['dimensions'][0]['id'],
                            'crashes': x['metrics'][0],
                            'p_device': x['metrics'][4]
                        } if
                            x['dimensions'][0]['method_name'] is not None
                        else None,
                    data)))

    if save_to_file:
        if extension is not None:
            file_name = f'{version}_{extension}_crash_stat.json'
        else:
            file_name = f'{version}_crash_stat.json'
        save_data_to_json_file(file_name, result)
    return result


def left_join_crash_stat(first_version, second_version) -> None:
    import pandas as pd
    pd.set_option('display.max_colwidth', 500)

    first_version_df = pd.DataFrame(first_version)
    first_version_df = first_version_df[['id', 'title', 'crashes', 'p_device']]

    second_version_df = pd.DataFrame(second_version)
    second_version_df = second_version_df[['id', 'title', 'crashes', 'p_device']]

    res = pd.merge(first_version_df, second_version_df, left_on='id', right_on='id', how='left')

    res.to_html(f'left_join{first_version}-{second_version}.html')


def format_crash_free_data_for_uploading_to_stat(data: Dict, version, extension, save_to_file: bool = False):
    result = []

    pattern = '^(\d+)(\d{3})(\d)$'
    splitted_string = re.split(pattern, version)
    version = f'{splitted_string[1]}.{splitted_string[2].lstrip("0")}.{splitted_string[3]}'

    if extension is None:
        extension = 'all'
    elif extension == 'ru.yandex.mail.notificationcontentextension':
        extension = 'nce'
    elif extension == 'ru.yandex.mail.notificationserviceextension':
        extension = 'nse'
    else:
        raise ValueError('Unknown extension name')

    for i in range(0, len(data['data'][0]['metrics'][0])):
        a = {
            'fielddate': data['time_intervals'][i][0],
            'version': version,
            'crashes_free_sessions': data['data'][0]['metrics'][0][i],
            'crashes_free_devices': data['data'][0]['metrics'][1][i],
            'extension': extension
        }
        result.append(a)

    if save_to_file:
        if extension is not None:
            file_name = f'stat_{version}_{extension}_crash_free_sessions_and_devices.json'
        else:
            file_name = f'stat_{version}_crash_free_sessions_and_devices.json'
        save_data_to_json_file(file_name, result)
    print(result)
    return result


def format_crash_count_by_period_data_for_uploading_to_stat(data: Dict, version: str, extension: str, save_to_file: bool = False):
    result = []

    if version is not None:
        pattern = '^(\d+)(\d{3})(\d)$'
        splitted_string = re.split(pattern, version)
        version = f'{splitted_string[1]}.{splitted_string[2].lstrip("0")}.{splitted_string[3]}'
    else:
        version = 'all'

    if extension is None:
        extension = 'all'
    elif extension == 'ru.yandex.mail.notificationcontentextension':
        extension = 'nce'
    elif extension == 'ru.yandex.mail.notificationserviceextension':
        extension = 'nse'
    else:
        raise ValueError('Unknown extension name')

    for i in range(0, len(data['data'][0]['metrics'][0])):
        a = {
            'fielddate': data['time_intervals'][i][0],
            'version': version,
            'crashes': data['data'][0]['metrics'][0][i],
            'crashed_devices': data['data'][0]['metrics'][1][i],
            'p_crashed_devices': data['data'][0]['metrics'][2][i],
            'extension': extension
        }
        result.append(a)
    if save_to_file:
        file_name = f'stat_all_{version}_{extension}_crashes_count_by_period.json' if extension is not None else f'stat_all_{version}_crashes_count_by_period.json'
        save_data_to_json_file(file_name, result)
    print(result)
    return result


def upload_to_stat_crash_count_by_period_data(table: str, start_date: str, end_date: str, save_to_file: bool = False):
    report = stat_client.get_report(table)
    for version in [None, '40010', '40020', '40030', '40040', '40050', '40051', '40060', '40061', '40063', '40070']:
        for extension in [None, Extension.nce, Extension.nse]:
            crashes_count_by_period = client.get_crashes_count_and_device_percent_by_period(start_date=start_date,
                                                                                            end_date=end_date,
                                                                                            app_version=version,
                                                                                            extension=extension)
            if save_to_file:
                file_name = f'all_{version}_{extension}_crashes_count_by_period.json' if extension is not None else f'all_{version}_crashes_count_by_period.json'
                save_data_to_json_file(file_name, crashes_count_by_period)

            data = format_crash_count_by_period_data_for_uploading_to_stat(crashes_count_by_period, version, extension, save_to_file=save_to_file)
            report.upload_data(scale="daily", data=data)


def upload_to_stat_crash_free_data(table: str, save_to_file: bool = False) -> None:
    report = stat_client.get_report(table)
    for version in CONFIG.keys():
        for extension in [None, Extension.nce, Extension.nse]:
            crash_free_sessions_and_devices = client.get_crashes_free_session_and_devices(start_date=CONFIG[version]['start_date'],
                                                                                          end_date=CONFIG[version]['end_date'],
                                                                                          app_version=version,
                                                                                          extension=extension)
            if save_to_file:
                if extension is not None:
                    file_name = f'{version}_{extension}_crash_free_sessions_and_devices.json'
                else:
                    file_name = f'{version}_crash_free_sessions_and_devices.json'
                save_data_to_json_file(file_name, crash_free_sessions_and_devices)

            data = format_crash_free_data_for_uploading_to_stat(crash_free_sessions_and_devices,
                                                                version,
                                                                extension,
                                                                save_to_file=save_to_file)
            report.upload_data(scale="daily", data=data)


def get_crash_stat_and_format(save_to_file: bool = False):
    for version in CONFIG.keys():
        for extension in [None, Extension.nce, Extension.nse]:
            crash_list = client.get_crash_stat(start_date=CONFIG[version]['start_date'],
                                               end_date=CONFIG[version]['end_date'],
                                               limit=1000,
                                               app_version=version,
                                               extension=extension)
            formatted_data = format_crash_data(crash_list, version, extension)
            if save_to_file:
                file_name = f'{version}_{extension}_crashes_stat.json' if extension is not None else f'{version}_crashes_stat.json'
                save_data_to_json_file(file_name, crash_list)


if __name__ == '__main__':
    client = AppMetricaClient(os='iOS', auth=OAUTH_APPMETRICA, api_key='29733')
    upload_to_stat_crash_count_by_period_data(table="Adhoc/fanem/optimization/crashes_per_version_test",
                                              start_date='2019-10-15',
                                              end_date='2020-02-03')
    upload_to_stat_crash_free_data(table="/Adhoc/fanem/optimization/crash_free_test")
