# coding=utf-8
import argparse
import json
import re
import sys

import requests
import datetime
from startrek_client import Startrek

UNKNOWN = '437'
RESOLUTION_FIXED_ID = '1'
RESOLUTION_WONTFIX_ID = '2'

DATE_FORMAT = "%Y-%m-%d"
STARTREK_API_PATH = "https://st-api.yandex-team.ru/v2"


# status should be one of "OPEN", "CLOSED", "SEALED"
def change_crashgroup_status(app_id, crash_group_id, token, status):
    url = 'https://api.appmetrica.yandex.ru/management/v1/application/{app_id}/crashes/{crash_group_id}/status'.format(
        app_id=app_id,
        crash_group_id=crash_group_id
    )
    resp = requests.post(url, headers={
        'Authorization': 'OAuth {}'.format(token)
    }, data=json.dumps({
        'value': status
    })
                         )
    if resp.status_code == 200:
        print("status for crashgroup {} changed to {}".format(crash_group_id, status))
    else:
        print("Failed to add comment: status {}, msg:".format(resp.status_code, resp.content))

def add_crashgroup_comment(app_id, crash_group_id, token, comment):
    url = 'https://api.appmetrica.yandex.ru/management/v1/application/{app_id}/crashes/{crash_group_id}/comment'.format(
        app_id=app_id,
        crash_group_id=crash_group_id
    )
    resp = requests.post(url, headers={
                             'Authorization': 'OAuth {}'.format(token)
                         }, data=json.dumps({
                             'value': comment
                         })
    )
    if resp.status_code == 200:
        print("Comment added for crashgroup {}".format(crash_group_id))
    else:
        print("Failed to add comment for crashgroup: status {}, msg:".format(crash_group_id, resp.status_code, resp.content))


backoff_config_dict = {
    "14836": {
        "name": "Android Mail",
        "platform": "Android",
        "startrek_queue": "MOBILEMAIL",
        "startrek_component": "Android",
        "followers": ["olyd", "arovkova"],
        "priorities": [
            {
                "priority": "critical",
                "weight": 100,
                "affected_devices_threshold_per_day": 500,
                "total_crashes_per_day": 1000,
                "affected_devices_percent_per_day": 0.05
            },
            {
                "priority": "normal",
                "weight": 50,
                "affected_devices_threshold_per_day": 250,
                "total_crashes_per_day": 500,
                "affected_devices_percent_per_day": 0.025
            },
            {
                "priority": "minor",
                "weight": 1,
                "affected_devices_threshold_per_day": 125,
                "total_crashes_per_day": 250,
                "affected_devices_percent_per_day": 0.013
            }
        ]
    },
    "29733": {
        "name": "Android Mail",
        "platform": "iOS",
        "startrek_queue": "MOBILEMAIL",
        "startrek_component": "iOS",
        "followers": ["fanem", "odango"],
        "priorities": [
            {
                "priority": "critical",
                "weight": 100,
                "affected_devices_threshold_per_day": 500,
                "total_crashes_per_day": 1000,
                "affected_devices_percent_per_day": 0.05
            },
            {
                "priority": "normal",
                "weight": 50,
                "affected_devices_threshold_per_day": 250,
                "total_crashes_per_day": 500,
                "affected_devices_percent_per_day": 0.025
            },
            {
                "priority": "minor",
                "weight": 1,
                "affected_devices_threshold_per_day": 125,
                "total_crashes_per_day": 250,
                "affected_devices_percent_per_day": 0.013
            }
        ]
    }
}


def find_component(st_client, queue,  component):
    """
    Finds components in Startrek by name

    :param st_client: Startrek client
    :param queue: queue name in startrek
    :param component: component name to find
    :return: Component instance or list of Component instance
    """
    client_components = st_client.queues[queue].components
    components_to_filter = list(filter(lambda x: x.name == component, client_components))
    return components_to_filter[0] if len(components_to_filter) > 0 else None


def find_ticket_by_crashgroup_id(startrek_client, startrek_queue, crashgroup_id):
    filter_dict = {
        'queue': startrek_queue,
        'crashId': crashgroup_id
    }
    issues = startrek_client.issues.find(
        filter=filter_dict,
        per_page=100
    )
    if len(issues) == 0:
        return None
    return issues[0]


def get_start_end_timestamps(cli_args):
    date_str = cli_args.date
    if date_str is None:
        start_date_obj = datetime.datetime.today().date() - datetime.timedelta(days=2)
        end_date_obj = datetime.datetime.today().date() - datetime.timedelta(days=1)
    else:
        end_date_obj = datetime.datetime.strptime(date_str, DATE_FORMAT)
        start_date_obj = end_date_obj.date() - datetime.timedelta(days=1)

    return start_date_obj.strftime(DATE_FORMAT), end_date_obj.strftime(DATE_FORMAT)


def get_config_dict(config_file):
    config_dict = backoff_config_dict
    try:
        with open(config_file, 'r') as config_file_json:
            config_dict = json.loads(config_file_json.read())
    except:
        print("could not parse json with config file!")
        pass
    return config_dict


def is_crashgroup_above_threshold(config, total_crashes_abs, devices_crashes_percent, crashed_devices_abs):
    return total_crashes_abs >= config['total_crashes_per_day'] and \
           crashed_devices_abs >= config['affected_devices_threshold_per_day'] and \
           devices_crashes_percent >= config['affected_devices_percent_per_day']


def get_crashgroup_priority(config, total_crashes_abs, devices_crashes_percent, crashed_devices_abs):
    result_priority = None
    priorities_config = config['priorities']
    sorted_priorities = sorted(priorities_config, key=lambda x: x['weight'])
    for priority_item in sorted_priorities:
        if total_crashes_abs >= priority_item['total_crashes_per_day'] and \
           crashed_devices_abs >= priority_item['affected_devices_threshold_per_day'] and \
           devices_crashes_percent >= priority_item['affected_devices_percent_per_day']:
           result_priority = priority_item['priority']
    return result_priority


def find_fix_version_object(startrek_client, queue_name, version_name):
    all_versions = startrek_client.queues[queue_name].versions
    return list(filter(lambda x: x.name == version_name, all_versions))


def get_max_version_from_all_version(startrek_client, queue_name, component_name):
    all_versions = startrek_client.queues[queue_name].versions
    filtered_version = list(filter(lambda x: x.archived == False and component_name in x.name, all_versions))
    versions = list(
        map(lambda x: sum(int(x[i]) * pow(10, 3 * (len(x) - i)) for i in range(len(x))),
            map(lambda x: x.split('.'),
                map(lambda x: str(x[0]),
                    filter(
                        lambda x: len(x) > 0,
                        map(
                            lambda x: re.findall("[0-9]+\\.[0-9]+\\.[0-9]+", x.name),
                            filtered_version
                        )
                    )
                    )
                )
            )
    )
    versions.sort(reverse=True)
    return versions[0]


def create_ticket_internal(component, component_objs, crashes_percent, crashgroup_id, desc, name, startrek_client,
                           startrek_queue, version, ticket_priority_key, followers):
    affected_version = find_fix_version_object(startrek_client, startrek_queue, '{} {}'.format(component, version))
    fix_version_obj = get_max_version_from_all_version(startrek_client, startrek_queue, component)
    users = startrek_client.users
    return startrek_client.issues.create(
        queue=startrek_queue,
        summary=name,
        type=startrek_client.issue_types['bug'],
        description=desc,
        crashId=crashgroup_id,
        crash='Yes',
        tags=[u'event_skip', u'inProd'],
        bugType=u'Метрика',
        affectedVersions=affected_version,
        fixVersions=fix_version_obj,
        audience=str(crashes_percent),
        components=[component_objs],
        stage=u'Production',
        priority=startrek_client.priorities[ticket_priority_key].id,
        followers=list(map(lambda x: users[x], followers)),
        errorSource=u'Fresh Look',
        reproducibility=u'Периодически',
        howReproduce=u'Not reproduce by team',
        CauseOfMissingABug=UNKNOWN
    )


def prepare_ticket_description(crashes_percent, crashgroup_url, devices_percent, crashes_abs, first, last):
    desc = '''
**Креш-группа превысила пороги: {}**\n
процент крешей: {}\n
процент девайсов: {}\n
крешей за день: {}\n
Обнаружен в версии: {}\n
Последнее воспроизведение: {}\n
'''.format(crashgroup_url, crashes_percent, devices_percent, crashes_abs, first, last)
    return desc

def get_weight_by_priority_from_config(config, priority_name):
    priorities_config = config['priorities']
    sorted_priorities = sorted(priorities_config, key=lambda x: x['weight'])
    for priority_item in sorted_priorities:
        if priority_item['priority'] == priority_name:
            return priority_item['weight']
    return sorted_priorities[0]['weight']

def main():
    parser = argparse.ArgumentParser(description='Get crashes for specified platform, APIKey')
    parser.add_argument('--app_metrika_token', required=True, help='AppMetrica OAuth Token')
    parser.add_argument('--startrek_token', required=True, help='Startrek OAuth Token')
    parser.add_argument('--api_key', required=True, help='APIKey for application')
    parser.add_argument('--config_file', required=False, help='Json file with config')
    parser.add_argument('--limit', required=False, default=1000, help='Number of crashgroups to obtain from appmetrica')
    parser.add_argument('--noop', dest='noop', action='store_true')
    parser.add_argument('--date', required=False, help='Date in format YYYY-MM-DD, default yesterday')
    parser.set_defaults(noop=False)
    args = parser.parse_args()
    is_noop = args.noop
    api_key = args.api_key
    metrika_auth_token = args.app_metrika_token
    startrek_auth_token = args.startrek_token
    config_file = args.config_file
    config_dict = get_config_dict(config_file)[api_key]
    startrek_queue = config_dict['startrek_queue']
    limit = args.limit
    start_date_str, end_date_str = get_start_end_timestamps(args)
    res = get_appmetrika_response(api_key, metrika_auth_token, end_date_str, limit, start_date_str)

    if res.status_code == 200:
        responseMap = json.loads(res.content)
        if u'query' in responseMap and u'metrics' in responseMap['query']:
            rows_dict = convert_response_to_rows_dict(responseMap)
            startrek_client = Startrek(useragent="MyAgent", base_url=STARTREK_API_PATH, token=startrek_auth_token)
            for row in rows_dict:
                crashgroup_id = row['id']
                existing_ticket = find_ticket_by_crashgroup_id(startrek_client, startrek_queue, crashgroup_id)
                component = config_dict['startrek_component']
                followers = config_dict['followers']
                platform = str(config_dict['platform']).lower()
                component_objs = find_component(startrek_client, startrek_queue,
                                                'Crash monitoring {}'.format(component))
                if existing_ticket is None:
                    print('ticket for crashgroup {} was not found'.format(crashgroup_id))
                    first_occurrence = row['first_occurrence']
                    last_occurrence = row['last_occurrence']
                    version_and_build = first_occurrence.split(" ") if first_occurrence else ''
                    version = version_and_build[0] if len(version_and_build) > 0 else 'unknown'
                    name = '[CRASH_MONITOR] {}'.format(row['name'])
                    crashes_percent = row['norm(ym:cr2:crashes)']
                    devices_percent = row['ym:cr2:crashesDevicesPercentage']
                    crash_devices_absolute = row['ym:cr2:crashDevices']
                    total_crashes_absolute = row['ym:cr2:crashes']
                    print('crashgroup {}: crashes_percent = {}, devices_percent = {}, crash_devices_absolute = {}, total_crashes_absolute = {}'.format(
                        crashgroup_id, crashes_percent, devices_percent, crash_devices_absolute, total_crashes_absolute)
                    )
                    ticket_priority = get_crashgroup_priority(config_dict, total_crashes_absolute, devices_percent, crash_devices_absolute)
                    if ticket_priority:
                        print('crashgroup {}, priority: {}'.format(crashgroup_id, ticket_priority))
                        print('Ticket for crashgroup {} is critical level, creating new ticket in startrek'.format(crashgroup_id))
                        crashgroup_url = 'https://appmetrica.yandex.ru/statistic?appId={api_key}&report=crash-logs-{platform}&sampling=1&crashGroupId={crashgroup_id}'.format(
                            platform=platform,
                            api_key=api_key,
                            crashgroup_id=crashgroup_id
                        )
                        print('crashgroup url: {}'.format(crashgroup_url))
                        # prepare description
                        desc = prepare_ticket_description(
                            crashes_percent,
                            crashgroup_url,
                            devices_percent,
                            crash_devices_absolute,
                            first_occurrence,
                            last_occurrence
                        )
                        # create ticket
                        print('Creating ticket for crashgroup {}'.format(crashgroup_id))
                        if not is_noop:
                            issue = create_ticket_internal(
                                component,
                                component_objs,
                                crashes_percent,
                                crashgroup_id,
                                desc,
                                name,
                                startrek_client,
                                startrek_queue,
                                version,
                                ticket_priority,
                                followers
                            )
                            print('Created ticket {} for crashgroup {}'.format(issue.key, crashgroup_id))
                            # add comment in appmetrika for crashgroup
                            print('Adding comment in appmetrika crashgroup {} with link to ticket {}'.format(crashgroup_id, issue.key))
                            if not is_noop:
                                add_crashgroup_comment(api_key, crashgroup_id, metrika_auth_token, 'https://st.yandex-team.ru/{}'.format(issue.key))
                            print('Comment added')
                else:
                    print('Ticket {} for crashgroup {} was found'.format(existing_ticket.key, crashgroup_id))
                    print('Checking if update in priority is needed..')
                    current_priority = str(existing_ticket.priority.key)
                    current_priority_weight = get_weight_by_priority_from_config(config_dict, current_priority)
                    current_state = str(existing_ticket.status.key)
                    crashes_percent = row['norm(ym:cr2:crashes)']
                    devices_percent = row['ym:cr2:crashesDevicesPercentage']
                    crash_devices_absolute = row['ym:cr2:crashDevices']
                    total_crashes_absolute = row['ym:cr2:crashes']
                    first_occurrence = row['first_occurrence']
                    last_occurrence = row['last_occurrence']
                    crashgroup_url = 'https://appmetrica.yandex.ru/statistic?appId={api_key}&report=crash-logs-android&sampling=1&crashGroupId={crashgroup_id}'.format(
                        api_key=api_key,
                        crashgroup_id=crashgroup_id
                    )
                    new_ticket_priority = get_crashgroup_priority(config_dict, total_crashes_absolute, devices_percent,
                                                              crash_devices_absolute)
                    new_ticket_priority_weight = get_weight_by_priority_from_config(config_dict, new_ticket_priority)
                    if current_state.lower() == 'closed':
                        desc = prepare_ticket_description(
                            crashes_percent,
                            crashgroup_url,
                            devices_percent,
                            crash_devices_absolute,
                            first_occurrence,
                            last_occurrence
                        )
                        if new_ticket_priority:
                            if new_ticket_priority_weight != current_priority_weight and \
                                    existing_ticket.resolution.id == RESOLUTION_FIXED_ID:
                                print('Reopening issue(priority changed) {}..'.format(existing_ticket.key))
                                if not is_noop:
                                    existing_ticket.transitions['reopen'].execute(
                                        comment=desc,
                                        priority=startrek_client.priorities[new_ticket_priority],
                                        components=[component_objs],
                                    )
                            else:
                                print('Reopening issue {}..'.format(existing_ticket.key))
                                if not is_noop:
                                    existing_ticket.transitions['reopen'].execute(
                                        priority=startrek_client.priorities[new_ticket_priority],
                                        components=[component_objs],
                                    )
                    else:
                        if new_ticket_priority_weight > current_priority_weight:
                            print('Updating priority for ticket {}..'.format(existing_ticket.key))
                            if not is_noop:
                                existing_ticket.update(priority=startrek_client.priorities[new_ticket_priority])
                    print('Adding comment in appmetrika crashgroup {} with link to ticket {}'.format(crashgroup_id, existing_ticket.key))
                    if not is_noop:
                        add_crashgroup_comment(api_key, crashgroup_id, metrika_auth_token, 'https://st.yandex-team.ru/{}'.format(existing_ticket.key))
                        change_crashgroup_status(api_key, crashgroup_id, metrika_auth_token, 'OPEN')
        print('Completed..')
        sys.exit(0)
    else:
        print('Failed to get response from Metrika! Metrika responded with status code {} and message {}'.format(res.status_code, res.content))
        sys.exit(1)


def convert_response_to_rows_dict(map):
    query_metrics = map[u'query'][u'metrics']
    metrics_with_indexes = dict()
    yt_rows_dicts = list()
    for idx, item in enumerate(query_metrics):
        metrics_with_indexes[idx] = item
    if u'data' in map:
        for data_item in map[u'data']:
            this_row_dict = dict()
            for idx, metric in enumerate(data_item[u'metrics']):
                this_row_dict[metrics_with_indexes[idx]] = metric
            if len(data_item[u'dimensions']) > 0:
                dimensions = data_item[u'dimensions'][0]
                for key in dimensions:
                    this_row_dict[key] = dimensions[key]
            yt_rows_dicts.append(this_row_dict)
    return yt_rows_dicts


def get_appmetrika_response(api_key, auth_token, end_date, limit, start_date):
    req = "https://api.appmetrica.yandex.ru/stat/v1/data?filters=crashGroupStatusSpecial=='OPEN'&id={api_key}&date1={start_date}&date2={end_date}&metrics=ym:cr2:crashes,norm(ym:cr2:crashes),ym:cr2:crashDevices,norm(ym:cr2:crashDevices),ym:cr2:crashesDevicesPercentage,ym:cr2:crashesUndecoded&dimensions=ym:cr2:crashGroupObj&sort=-ym:cr2:crashes&offset=1&limit={limit}&debug=json&accuracy=1&proposedAccuracy=true".format(
        api_key=api_key,
        limit=limit,
        start_date=start_date,
        end_date=end_date
    )
    res = requests.get(req, headers={'Authorization': 'OAuth {}'.format(auth_token)},
                       verify=False)
    return res


if __name__ == "__main__":
    main()
