# coding=utf-8
import os
import yt.wrapper as yt
from contants import FILTERED_CRASHGROUPS_PATH
from startrek_client import Startrek
from contants import STARTREK_API_PATH
import appmetrika_utils
import re

MOBILE_MAIL_VERSION_REGEX = u'[0-9]+\.[0-9]+\.[0-9]+'

class MobileMailVersion:
    def __init__(self, version_str):
        version_parts = version_str.split('.')
        self.__major = None
        self.__minor = None
        self.__revision = None
        if len(version_parts) > 0:
            self.__major = version_parts[0]
        if len(version_parts) > 1:
            self.__minor = version_parts[1]
        if len(version_parts) > 2:
            self.__revision = version_parts[2]

    @property
    def major(self):
        return self.__major

    @property
    def minor(self):
        return self.__minor

    @property
    def revision(self):
        return self.__revision

    def __repr__(self):
        return '{}.{}.{}'.format(self.__major, self.__minor, self.__revision)


def module_filter(module):
    module_name = getattr(module, '__name__', '')
    if 'numpy' in module_name:
        return False
    if 'yt_yson_bindings' in module_name:
        return False
    if 'hashlib' in module_name:
        return False
    if 'hmac' in module_name:
        return False

    module_file = getattr(module, '__file__', '')
    if not module_file:
        return False
    if module_file.endswith('.so'):
        return False

    return True


def compare_versions(version1, version2):
    if version1.major > version2.major:
        return 1
    elif version1.major < version2.major:
        return -1
    else:
        if version1.minor > version2.minor:
            return 1
        elif version1.minor < version2.minor:
            return -1
        else:
            if version1.revision > version2.revision:
                return 1
            elif version1.revision < version2.revision:
                return -1
            else:
                return 0


def find_component(st_client, queue,  component):
    """
    Finds components in Startrek by name

    :param st_client: Startrek client
    :param queue: queue name in startrek
    :param component: component name to find
    :return: Component instance or list of Component instance
    """
    client_components = st_client.queues[queue].components
    components_to_filter = filter(lambda x: x.name == component, client_components)
    return components_to_filter

def get_latest_fix_version(queue_name, startrek_token, components=None):
    """
    Infers fix version as maximum version of FixVersion for tickets in status == Testing

    :param components: list of components name to filter
    :param queue_name: name of queue in Startrek
    :param startrek_token: auth token
    :return: AndroidVersion instance with maximum fix version or None if nothing was found
    """
    startrek_client = Startrek(useragent="MyAgent", base_url=STARTREK_API_PATH, token=startrek_token)
    client_components = startrek_client.queues[queue_name].components
    filter_dict = {'queue': queue_name, 'status': 'Testing'}
    components_to_filter = filter(lambda x: x.name in components, client_components)
    if components is not None:
        filter_dict['components'] = components_to_filter
    issues = startrek_client.issues.find(
        filter=filter_dict,
        per_page=100
    )
    fix_versions = []
    for issue in issues:
        if len(issue.fixVersions) > 0:
            fix_versions.append(issue.fixVersions[0].name)
    fix_versions = map(
        lambda x: MobileMailVersion(x),
        list(
            map(
                lambda x: x[0],
                list(filter(lambda x: len(x) > 0, list(map(lambda x: re.findall(MOBILE_MAIL_VERSION_REGEX, x), fix_versions))))
            )
        )
    )
    fix_versions.sort(cmp = compare_versions, reverse=True)
    return fix_versions[0] if len(fix_versions) > 0 else None


def find_fix_version_object(startrek_client, queue_name, version_name):
    all_versions = startrek_client.queues[queue_name].versions
    return list(filter(lambda x: x.name == version_name, all_versions))


def find_ticket_by_crashgroup_id(startrek_client, startrek_queue, crashgroup_id):
    filter_dict = {
        'queue': startrek_queue,
        'crashId': crashgroup_id
    }
    issues = startrek_client.issues.find(
        filter=filter_dict,
        per_page=100
    )
    if len(issues) == 0:
        return None
    return issues[0]


def create_ticket(app_id, startrek_token, yt_token, appmetrika_token, root_path):
    yt.config["token"] = yt_token
    yt.config["proxy"]["url"] = "hahn.yt.yandex.net"
    yt.config['pickling']['dynamic_libraries']['enable_auto_collection'] = True
    yt.config['pickling']['module_filter'] = module_filter
    path = os.path.join(root_path, 'metadata')
    table_row = yt.read_table(yt.TablePath(path))
    for row in table_row:
        if row['api_key'] == app_id:
            component = row['component']
            startrek_queue = row['startrek_queue']
            # read rows from filtered crashes table and for each create ticket
            startrek_client = Startrek(useragent="MyAgent", base_url=STARTREK_API_PATH,
                                       token=startrek_token)
            component_objs = find_component(startrek_client, startrek_queue, component)
            filtered_crashes_rows = yt.read_table(os.path.join(root_path, FILTERED_CRASHGROUPS_PATH))
            for single_crash in filtered_crashes_rows:
                crashgroup_id = single_crash['id']
                already_existed_ticket = find_ticket_by_crashgroup_id(startrek_client, startrek_queue,  crashgroup_id)
                if already_existed_ticket is not None:
                    # TODO this is case when in past days we found crash, need to check % of affected users/devices
                    pass

                first_occurrence = single_crash['first_occurrence']
                version_and_build = first_occurrence.split(" ")
                version=version_and_build[0] if len(version_and_build) > 0 else 'unknown'
                build=version_and_build[1] if len(version_and_build)> 1 else 'unknown'
                name = single_crash['name']
                crashes_percent = single_crash['norm(ym:cr2:crashes)']
                devices_percent = single_crash['ym:cr2:crashesDevicesPercentage']
                crashgroup_url='https://appmetrica.yandex.ru/statistic?appId={app_id}&report=crash-logs-android&sampling=1&crashGroupId={crashgroup_id}}'.format(
                    app_id=app_id,
                    crashgroup_id=crashgroup_id
                )

                desc = prepare_ticket_description(build, crashes_percent, crashgroup_url, devices_percent)
                # create ticket
                issue = create_ticket_internal(component, component_objs, crashes_percent, crashgroup_id, desc, name,
                                               startrek_client, startrek_queue, startrek_token, version)
                # add comment in appmetrika for crashgroup
                appmetrika_utils.add_comment(app_id, crashgroup_id, appmetrika_token, 'https://st.yandex-team.ru/{}'.format(issue.key))


def create_ticket_internal(component, component_objs, crashes_percent, crashgroup_id, desc, name, startrek_client,
                           startrek_queue, startrek_token, version):
    return startrek_client.issues.create(
        queue=startrek_queue,
        summary=name,
        type={'name': 'Bug'},
        description=desc,
        crashId=crashgroup_id,
        crash='Yes',
        tags=[u'event_skip'],
        bugType=u'Метрика',
        affectedVersions=[find_fix_version_object(startrek_client, startrek_queue, '{} {}'.format(component, version))],
        fixVersions=get_latest_fix_version(startrek_queue, startrek_token, component),
        audience=str(crashes_percent),
        components=component_objs,
        stage=u'Production'
    )


def prepare_ticket_description(build, crashes_percent, crashgroup_url, devices_percent):
    desc = '''
**Креш-группа превысила пороги: {}**
процент крешей: {}
процент девайсов: {}
Билд: {}
'''.format(crashgroup_url, crashes_percent, devices_percent, build)
    return desc
