from dataclasses import fields
from typing import List, Dict
from datetime import datetime, timedelta
from tp_api_client.tp_api_client import TestPalmClient
from tracker_api_client.tracker_client import TrackerClient
import logging
from yql.api.v1.client import YqlClient
from helpers import datetime_to_date_str, datetime_to_millis_str
from constants import OAUTH_TOKEN_TESTPALM, OAUTH_TOKEN_YT, LoggerConfig, OAUTH_TOKEN_STARTREK, Query, YtTablePath, \
    OAUTH_TOKEN_YQL, YqlQuery
from objects import Project, TestCase, YTClient, TestpalmExpression, Queue, RemoteLink, Issue, LinkedBugElement, \
    TestCaseElement

logger = logging.getLogger(LoggerConfig.name)
logging.basicConfig(format=LoggerConfig.format, level=logging.INFO)


def get_testcases_resolutions_and_write_to_table(yt_client, testpalm_client, start_dt: datetime, end_dt: datetime):
    projects = fields(Project)
    testcase_list = []

    def get_testruns(client, project_name: str, start_dt: datetime, end_dt: datetime) -> List[Dict]:
        testruns = []
        start = start_dt
        end = start_dt + timedelta(days=1)
        while end <= end_dt:
            expression = TestpalmExpression.testruns_by_period(datetime_to_millis_str(start),
                                                               datetime_to_millis_str(end))
            logger.info(f'Get testruns for project {project_name} with expression "{expression}"')
            testrun = client.get_testruns(project=project_name, include='id,testGroups', expression=expression)
            testruns.extend(testrun)
            logger.info('Testruns received successfully')
            start = end
            end = end + timedelta(days=1)
        return testruns

    for project in projects:
        project_name = project.default
        testruns = get_testruns(testpalm_client, project_name=project_name, start_dt=start_dt, end_dt=end_dt)
        logger.info(f'Get project definitions')
        definitions = testpalm_client.get_project_definitions(project=project_name)

        for testrun in testruns:
            for testgroup in testrun['testGroups']:
                for testcase in testgroup['testCases']:
                    tc = TestCase(project_name, definitions, testcase)
                    testcase_list.append(tc.convert_to_upload_data_format(testrun['id']))

    yt_client.schema = TestCaseElement
    yt_client.table_path = YtTablePath.case_status_in_run
    yt_client.create_table_if_needed_and_write(testcase_list)


def get_linked_to_issue_cases_and_write_to_table(yt_client, st_client, yql_client, start_ts: str, end_ts: str):
    def get_linked_to_issues_cases(client, queue: str) -> List[Dict]:
        def get_linked_to_issue_cases(links: List[Dict], issue_id: str) -> List[Dict]:
            result: List = []
            for link in links:
                link = RemoteLink(link)
                if link.is_testpalm_testcase_link():
                    issue = Issue(client.get_issue(issue_id))
                    result.append(LinkedBugElement.create(link, issue))
            return result

        query = Query.get_bugs_open_with_resolution_or_closed.format(queue=queue, start_ts=start_ts, end_ts=end_ts)
        logger.info(f'Find issues in queue "{queue}" by query "{query}"')
        issues = client.find_issues_by_query(query)
        logger.info('Get linked to issues cases')
        linked_to_issues_cases = [get_linked_to_issue_cases(client.get_remote_links(issue_id), issue_id) for issue_id in issues]
        logger.info('Linked cases received successfully')
        return [linked_to_issue_case for linked_to_issue_cases in linked_to_issues_cases for linked_to_issue_case in
                linked_to_issue_cases]

    def delete_duplicated_rows(yql_client, table_path: str):
        query = YqlQuery.delete_duplicates.format(table_path=table_path)
        request = yql_client.query(query, syntax_version=1)
        result = request.run()
        logging.info(f'Started YQL: https://yql.yandex-team.ru/Operations/{request.operation_id}')
        result.wait_progress()
        if result.status == 'ERROR':
            raise Exception('YQL finished with error!')

    queues = fields(Queue)
    links_list = []

    for queue in queues:
        queue_name = queue.default
        links = get_linked_to_issues_cases(st_client, queue_name)
        links_list.extend(links)

    yt_client.schema = LinkedBugElement
    yt_client.table_path = YtTablePath.linked_bugs
    yt_client.create_table_if_needed_and_write(links_list)
    delete_duplicated_rows(yql_client, YtTablePath.linked_bugs)


if __name__ == '__main__':
    yql_client = YqlClient(token=OAUTH_TOKEN_YQL)
    yt_client = YTClient(auth=OAUTH_TOKEN_YT)
    testpalm_client = TestPalmClient(auth=OAUTH_TOKEN_TESTPALM)
    st_client = TrackerClient(auth=OAUTH_TOKEN_STARTREK)

    # start_dt = datetime(2022, 6, 1)
    # end_dt = datetime(2022, 6, 19)
    start_dt = (datetime.today() - timedelta(weeks=1)).replace(hour=0, minute=0, second=0, microsecond=0)
    end_dt = datetime.today().replace(hour=0, minute=0, second=0, microsecond=0)

    get_testcases_resolutions_and_write_to_table(yt_client,
                                                 testpalm_client,
                                                 start_dt,
                                                 end_dt)
    get_linked_to_issue_cases_and_write_to_table(yt_client,
                                                 st_client,
                                                 yql_client,
                                                 datetime_to_date_str(start_dt),
                                                 datetime_to_date_str(end_dt))
