import logging
from dataclasses import dataclass, asdict, fields
from typing import Any, Dict, List

from tp_api_client.tp_api_client import Operators, RunStatus

from constants import LoggerConfig
from helpers import str_to_millis_with_format
import yt.wrapper as yt

logger = logging.getLogger(LoggerConfig.name)
logging.basicConfig(format=LoggerConfig.format, level=logging.INFO)


@dataclass(frozen=True)
class Project:
    mobilemail: str = 'mobilemail'
    adisk: str = 'adisk'
    idisk: str = 'idisk'


@dataclass(frozen=True)
class Queue:
    mobmail_ios: str = 'MOBMAILIOS'
    mobmail_android: str = 'MOBMAILANDROID'
    mobdisk_ios: str = 'MOBDISKIA'
    mobdisk_android: str = 'MOBDISKAA'
    # mobtelemost: str = 'MOBTELA'
    # webdisk: str = 'DISKEXP'
    # webdisk_client: str = 'DISKCLIENTEXP'
    # webmail: str = 'MAILEXP'
    # podisk: str = 'DISCSWASESSORS'


@dataclass(frozen=True)
class CasePriority:
    acceptance: str = 'acceptance'
    business_logic: str = 'business_logic'
    regress: str = 'regress'
    full_regress: str = 'full_regress'
    once: str = 'once'
    smoke: str = 'smoke'
    unknown: str = 'unknown'


@dataclass(frozen=True)
class CaseImportance:
    high: str = 'High'
    medium: str = 'Medium'
    low: str = 'Low'
    zero: str = 'Zero'
    smoke: str = 'Smoke'


@dataclass(frozen=True)
class FeaturePriority:
    high: str = 'High'
    medium: str = 'Medium'
    low: str = 'Low'


@dataclass(frozen=True)
class RunTestcaseStatus:
    blocked: str = 'BLOCKED'
    broken: str = 'BROKEN'
    created: str = 'CREATED'
    failed: str = 'FAILED'
    knownbug: str = 'KNOWNBUG'
    passed: str = 'PASSED'
    skipped: str = 'SKIPPED'
    started: str = 'STARTED'


@dataclass
class TestCaseElement:
    started_time: int
    finished_time: int
    project: str
    testrun_id: str
    testcase_id: int
    priority: str
    testcase_status: str
    started_by: str
    finished_by: str
    testcase_uuid: str
    is_autotest: bool
    is_honeypot: bool


@dataclass(frozen=True)
class ExpressionElement:
    type: str
    left: Any
    right: Any


@dataclass(frozen=True)
class ExpressionCondition:
    type: str
    key: str
    value: str


project_to_field_name = {
    Project.mobilemail: {
        'case_importance': 'Case Importance',
        'feature_priority': 'Feature Priority'
    },
    Project.idisk: {
        'case_importance': 'testcase_priority',
        'feature_priority': 'feature_priority'
    },
    Project.adisk: {
        'case_importance': 'testcase_priority',
        'feature_priority': 'feature_priority'
    }
}


class Issue:
    def __init__(self, data):
        self.__id = data.key
        self.__status = data.status.key
        self.__resolution = data.resolution.key.replace("'", '')
        self.__created_at = data.createdAt
        self.__updated_at = data.updatedAt
        self.__created_by = data.createdBy.id

    @property
    def id(self) -> str:
        return self.__id

    @property
    def status(self) -> str:
        return self.__status

    @property
    def resolution(self) -> str:
        return self.__resolution

    @property
    def created_at_in_millis(self) -> int:
        return str_to_millis_with_format(self.__created_at, '%Y-%m-%dT%H:%M:%S.%f%z')

    @property
    def updated_at_in_millis(self) -> int:
        return str_to_millis_with_format(self.__updated_at, '%Y-%m-%dT%H:%M:%S.%f%z')

    @property
    def created_by(self) -> str:
        return self.__created_by


class RemoteLink:
    def __init__(self, data: Dict):
        self.__issue_key = data['issue_key']
        self.__remote_service_type = data['remote_service_type']
        self.__remote_service_name = data['remote_service_name']

    def __get_testpalm_field(self, field: str) -> str:
        field_to_element_index = {'project': 1, 'testcase_id': 2}
        if field not in ['project', 'testcase_id']:
            raise ValueError(f'Unknown field. Field - {field}')
        if self.is_testpalm_testcase_link:
            return self.__issue_key.split('/')[field_to_element_index[field]]
        raise ValueError(f'This is not a testpalm link. Link - {self.__issue_key}')

    def __get_testpalm_testcase_id(self) -> int:
        return int(self.__get_testpalm_field('testcase_id'))

    def __get_testpalm_project(self) -> str:
        return self.__get_testpalm_field('project')

    def is_testpalm_link(self) -> bool:
        return self.__remote_service_name == 'TestPalm' and self.__remote_service_type == 'ru.yandex.testpalm'

    def is_testpalm_testcase_link(self) -> bool:
        splitted_issue_key = self.__issue_key.split('/')
        return self.is_testpalm_link() and splitted_issue_key[0] == 'testcases'

    def is_testpalm_testrun_link(self) -> bool:
        splitted_issue_key = self.__issue_key.split('/')
        return self.is_testpalm_link() and splitted_issue_key[0] == 'testrun'

    @property
    def testpalm_testcase_id(self) -> int:
        return self.__get_testpalm_testcase_id()

    @property
    def testpalm_project(self) -> str:
        return self.__get_testpalm_project()


@dataclass
class LinkedBugElement:
    testcase_id: int
    project: str
    st_issue_id: str
    st_issue_status: str
    st_issue_resolution: str
    st_issue_created_at: int
    st_issue_updated_at: int
    st_issue_created_by: str

    @staticmethod
    def create(link: RemoteLink, issue: Issue) -> dict:
        return asdict(LinkedBugElement(
            testcase_id=link.testpalm_testcase_id,
            project=link.testpalm_project,
            st_issue_id=issue.id,
            st_issue_status=issue.status,
            st_issue_resolution=issue.resolution,
            st_issue_created_at=issue.created_at_in_millis,
            st_issue_updated_at=issue.updated_at_in_millis,
            st_issue_created_by=issue.created_by
        ))


class TestCase:
    def __init__(self, project: str, definitions: List[Dict], data: Dict):
        self.__project = project
        self.__data = data
        self.__definitions = definitions
        self.__id = self.__data['testCase']['id']
        self.__feature_priority = self.get_attribute(project_to_field_name[project]['feature_priority'])
        self.__case_importance = self.get_attribute(project_to_field_name[project]['case_importance'])
        self.__case_priority = self.get_case_priority()
        self.__started_time = self.__data['startedTime']
        self.__finished_time = self.__data['finishedTime']
        self.__status = self.__data['status']
        self.__started_by = self.__data['startedBy']
        self.__finished_by = self.__data['finishedBy']
        self.__uuid = self.__data['uuid']
        self.__is_autotest = self.__data['testCase']['isAutotest']
        self.__is_honeypot = 'CH' in self.get_attribute('Case Type')

    def get_definition_id(self, name: str) -> str:
        return next((definition['id'] for definition in self.__definitions if definition['title'] == name), None)

    def get_attribute(self, name: str) -> List[str]:
        definition_id = self.get_definition_id(name)
        if definition_id in self.__data['testCase']['attributes']:
            return self.__data['testCase']['attributes'][definition_id]
        return []

    @property
    def feature_priority(self) -> List[str]:
        return self.__feature_priority

    @property
    def case_importance(self) -> List[str]:
        return self.__case_importance

    @property
    def case_priority(self) -> str:
        return self.__case_priority

    @property
    def started_time(self) -> int:
        return self.__started_time

    @property
    def finished_time(self) -> int:
        return self.__finished_time

    @property
    def id(self) -> int:
        return self.__id

    @property
    def status(self) -> str:
        return self.__status

    @property
    def started_by(self) -> str:
        return self.__started_by

    @property
    def finished_by(self) -> str:
        return self.__finished_by

    @property
    def uuid(self) -> str:
        return self.__uuid

    @property
    def project(self) -> str:
        return self.__project

    @property
    def is_autotest(self) -> bool:
        return self.__is_autotest

    @property
    def is_honeypot(self) -> bool:
        return self.__is_honeypot

    def get_case_priority(self) -> str:
        if self.__is_acceptance_case():
            return CasePriority.acceptance
        if self.__is_business_logic_case():
            return CasePriority.business_logic
        if self.__is_regress_case():
            return CasePriority.regress
        if self.__is_full_regress_case():
            return CasePriority.full_regress
        if self.__is_once_case():
            return CasePriority.once
        if self.__is_smoke_case():
            return CasePriority.smoke
        logger.warning(f'Unknown case priority.\n'
                       f'Case id: {self.id}\n'
                       f'Feature priority: {self.feature_priority},\n'
                       f'Case importance: {self.case_importance}\n')
        return CasePriority.unknown

    def __is_acceptance_case(self) -> bool:
        return FeaturePriority.high in self.feature_priority and CaseImportance.high in self.case_importance

    def __is_business_logic_case(self) -> bool:
        return (
            (FeaturePriority.high in self.feature_priority and CaseImportance.medium in self.case_importance)
            or
            (FeaturePriority.medium in self.feature_priority and CaseImportance.high in self.case_importance)
        )

    def __is_regress_case(self) -> bool:
        return (
            (FeaturePriority.high in self.feature_priority and CaseImportance.low in self.case_importance)
            or
            (FeaturePriority.medium in self.feature_priority and CaseImportance.medium in self.case_importance)
            or
            (FeaturePriority.low in self.feature_priority and CaseImportance.high in self.case_importance)
        )

    def __is_full_regress_case(self) -> bool:
        return (
            (FeaturePriority.medium in self.feature_priority and CaseImportance.low in self.case_importance)
            or
            (FeaturePriority.low in self.feature_priority and CaseImportance.medium in self.case_importance)
        )

    def __is_once_case(self) -> bool:
        return (
            (FeaturePriority.high in self.feature_priority and CaseImportance.zero in self.case_importance)
            or
            (FeaturePriority.medium in self.feature_priority and CaseImportance.zero in self.case_importance)
            or
            (FeaturePriority.low in self.feature_priority and CaseImportance.low in self.case_importance)
            or
            (FeaturePriority.low in self.feature_priority and CaseImportance.zero in self.case_importance)
        )

    def __is_smoke_case(self) -> bool:
        return CaseImportance.smoke in self.case_importance

    def convert_to_upload_data_format(self, testrun_id: str) -> Dict:
        return asdict(TestCaseElement(
            self.started_time,
            self.finished_time,
            self.project,
            testrun_id,
            self.id,
            self.case_priority,
            self.status,
            self.started_by,
            self.finished_by,
            self.uuid,
            self.is_autotest,
            self.is_honeypot
        ))


class YTClient:
    def __init__(self, auth, cluster='hahn'):
        self.__auth = auth
        self.__cluster = cluster
        self.__table_path = None
        self.__schema = None
        yt.config['proxy']['url'] = self.__cluster
        yt.config['token'] = self.__auth

    @property
    def cluster(self) -> str:
        return self.__cluster

    @cluster.setter
    def cluster(self, new_cluster: str) -> None:
        self.__cluster = new_cluster

    @property
    def table_path(self) -> str:
        return self.__table_path

    @table_path.setter
    def table_path(self, new_table_path: str) -> None:
        self.__table_path = new_table_path

    @property
    def schema(self) -> str:
        return self.__schema

    @schema.setter
    def schema(self, dto):
        dto_fields = fields(dto)
        python_type_to_yt_type: Dict[str, str] = {
            'int': 'int64',
            'str': 'string',
            'bool': 'boolean'
        }
        self.__schema = list(map(lambda x: {'name': x.name, 'type': python_type_to_yt_type[x.type.__name__]}, dto_fields))

    def create_table_if_needed_and_write(self, data):
        yt.create('table', path=self.table_path, attributes={"schema": self.schema}, ignore_existing=True)
        yt.write_table(yt.TablePath(self.table_path, append=True),
                       data,
                       format=yt.JsonFormat(attributes={"encode_utf8": False}))

    def read_table(self):
        return yt.read_table(yt.TablePath(self.table_path),
                             format=yt.JsonFormat(attributes={"encode_utf8": False}))


@dataclass(frozen=True)
class TestpalmExpression:
    @staticmethod
    def testruns_by_period(start_ts: str, end_ts: str) -> str:
        return str(asdict(
            ExpressionElement(
                'AND',
                ExpressionElement('AND',
                                  ExpressionCondition(Operators.gt, 'createdTime', start_ts),
                                  ExpressionCondition(Operators.lt, 'createdTime', end_ts)),
                ExpressionCondition(Operators.neq, 'status', RunStatus.created)))).replace("'", '"')

