import os
from dataclasses import dataclass, asdict

OAUTH_TOKEN_TESTPALM = os.environ['OAUTH_TOKEN_TESTPALM']
OAUTH_TOKEN_YT = os.environ['OAUTH_TOKEN_YT']
OAUTH_TOKEN_YQL = os.environ['OAUTH_TOKEN_YQL']


@dataclass(frozen=True)
class YtTablePath:
    table_path: str = '//home/mailfront/qa/fanem/autotests/'
    automation_status: str = table_path + 'automation_status'


@dataclass(frozen=True)
class YqlQuery:
    delete_today_rows: str = """
    USE hahn;
    PRAGMA yt.Pool = "sherlock";
    INSERT INTO `{table_path}` WITH TRUNCATE
    SELECT *
    FROM `{table_path}`
    WHERE `date` != {date};
    """


@dataclass
class AutomationData:
    date: int
    project: str
    priority: str
    case_status: str
    cnt: int
    testpalm_filter: str
    testpalm_filter_for_url: str


@dataclass(frozen=True)
class LoggerConfig:
    format: str = u'%(asctime)s [%(levelname)s] %(module)s: %(message)s'
    name: str = 'autotest_coverage_logger'


@dataclass(frozen=True)
class Project:
    web_mail: str = 'mail-liza'
    touch_mail: str = 'mail-touch'
    calendar: str = 'cal'
    mobile_mail_android: str = 'mobilemail'
    mobile_mail_ios: str = 'mobilemail'
    mobile_disk_android: str = 'adisk'
    mobile_disk_ios: str = 'idisk'
    tuning360: str = 'tuning360'
    disk_public: str = 'diskpublic'
    disk_notes: str = 'disknotes'
    disk_client: str = 'diskclient'
    telemost: str = 'telemost'
    disk_for_business: str = 'diskforbusiness'


@dataclass(frozen=True)
class CommonFilters:
    actual: str = '{total_cases}'
    automated: str = '{total_cases} AND isAutotest = true'
    cannot_be_automated: str = '{total_cases} AND isAutotest = false AND NOT({mark_to_automate}) AND NOT({can_be_automated})'
    no_need_to_automate: str = '{total_cases} AND isAutotest = false AND NOT({mark_to_automate}) AND {can_be_automated} AND {no_automation_required}'
    can_be_automated: str = '{total_cases} AND isAutotest = false AND NOT({mark_to_automate}) AND {can_be_automated}'
    need_to_mark_can_be_automated: str = '{total_cases} AND isAutotest = false AND {mark_to_automate}'


acceptance_filter = '("{feature_priority_key}" = "High" AND "{testcase_priority_key}" = "High")'
bl_filter = '''(
("{feature_priority_key}" = "High" AND "{testcase_priority_key}" = "Medium")
OR
("{feature_priority_key}" = "Medium" AND "{testcase_priority_key}" = "High")
)'''
regress_filter = '''(
("{feature_priority_key}" = "High" AND "{testcase_priority_key}" = "Low")
OR
("{feature_priority_key}" = "Medium" AND "{testcase_priority_key}" = "Medium")
OR
("{feature_priority_key}" = "Low" AND "{testcase_priority_key}" = "High")
)'''
full_regress_filter = '''(
("{feature_priority_key}" = "Medium" AND "{testcase_priority_key}" = "Low")
OR
("{feature_priority_key}" = "Low" AND "{testcase_priority_key}" = "Medium")
)'''
once_filter = '''(
(("{feature_priority_key}" = "High" OR "{feature_priority_key}" = "Medium") AND "{testcase_priority_key}" = "Zero")
OR
("{feature_priority_key}" = "Low" AND "{testcase_priority_key}" != "High" AND "{testcase_priority_key}" != "Medium")
)'''
no_regress_level_filter = '("{feature_priority_key}" = EMPTY OR "{testcase_priority_key}" = EMPTY)'


@dataclass(frozen=True)
class RegressLevelFilter:
    acceptance: str = '{base_filter} AND %s' % acceptance_filter
    business_logic: str = '{base_filter} AND %s' % bl_filter
    regress: str = '{base_filter} AND %s' % regress_filter
    full_regress: str = '{base_filter} AND %s' % full_regress_filter
    once: str = '{base_filter} AND %s' % once_filter
    total: str = '{base_filter}'


@dataclass
class ProjectData:
    total_cases: str
    feature_priority_key: str
    testcase_priority_key: str
    mark_to_automate: str
    can_be_automated: str
    no_automation_required: str


all_keys = {
    'web_mail': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "Вся Лиза (по фичам)" != "Административное" AND "Case Type" != "CH"',
        feature_priority_key='Приоритет фичи',
        testcase_priority_key='Приоритет кейса',
        mark_to_automate='"Автоматизация" = EMPTY',
        can_be_automated='"Автоматизация" != "Нельзя автоматизировать"',
        no_automation_required='"Автоматизация" = "Не надо автоматизировать"',
    )),
    'touch_mail': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "Фичи тачика" != "[Административное]"',
        feature_priority_key='Приоритет фичи',
        testcase_priority_key='Приоритет кейса',
        mark_to_automate='"Автоматизация" = EMPTY',
        can_be_automated='"Автоматизация" != "Нельзя автоматизировать"',
        no_automation_required='"Автоматизация" = "Не надо автоматизировать"',
    )),
    'calendar': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "Фичи календаря" != "Административное"',
        feature_priority_key='Приоритет фичи',
        testcase_priority_key='Приоритет кейса',
        mark_to_automate='"Автоматизация" = EMPTY',
        can_be_automated='"Автоматизация" != "Нельзя автоматизировать"',
        no_automation_required='"Автоматизация" = "Не надо автоматизировать"',
    )),
    'mobile_mail_android': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "Components" = "Android" AND "Internal Testcase" != "true" AND "Case Type" != "CH"',
        feature_priority_key='Feature Priority',
        testcase_priority_key='Case Importance',
        mark_to_automate='"Autotest" = EMPTY',
        can_be_automated='"Autotest" != "can\'t be automated"',
        no_automation_required='"Autotest" = "won\'t be automated"',
    )),
    'mobile_mail_ios': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "Components" = "iOS" AND "Internal Testcase" != "true" AND "Case Type" != "CH"',
        feature_priority_key='Feature Priority',
        testcase_priority_key='Case Importance',
        mark_to_automate='"Autotest" = EMPTY',
        can_be_automated='"Autotest" != "can\'t be automated"',
        no_automation_required='"Autotest" = "won\'t be automated"',
    )),
    'disk_public': asdict(ProjectData(
         total_cases='status = "actual" AND "Тип действия" != "Я.Другое" AND "Case Type" != "CH"',
         feature_priority_key='Feature priority',
         testcase_priority_key='Test case priority',
         mark_to_automate='"is_automation_possible" = EMPTY',
         can_be_automated='"is_automation_possible" != "false"',
         no_automation_required='"no_automation_required" = "true"'
    )),
    'tuning360': asdict(ProjectData(
         total_cases='status = "actual" AND "Тип действия" != "Я.Другое" AND "Case Type" != "CH"',
         feature_priority_key='Feature priority',
         testcase_priority_key='Test case priority',
         mark_to_automate='"is_automation_possible" = EMPTY',
         can_be_automated='"is_automation_possible" != "false"',
         no_automation_required='"no_automation_required" = "true"'
    )),
    'telemost': asdict(ProjectData(
         total_cases='status = "actual" AND "Тип действия" != "Я.Другое" AND "Case Type" != "CH"',
         feature_priority_key='Feature priority',
         testcase_priority_key='Test case priority',
         mark_to_automate='"is_automation_possible" = EMPTY',
         can_be_automated='"is_automation_possible" != "false"',
         no_automation_required='"no_automation_required" = "true"'
    )),
    'disk_for_business': asdict(ProjectData(
         total_cases='status = "actual" AND "Тип действия" != "Я.Другое" AND "Case Type" != "CH"',
         feature_priority_key='Feature priority',
         testcase_priority_key='Test case priority',
         mark_to_automate='"is_automation_possible" = EMPTY',
         can_be_automated='"is_automation_possible" != "false"',
         no_automation_required='"no_automation_required" = "true"'
    )),
    'disk_notes': asdict(ProjectData(
         total_cases='status = "actual" AND "Тип действия" != "Я.Другое" AND "Case Type" != "CH"',
         feature_priority_key='Feature priority',
         testcase_priority_key='Test case priority',
         mark_to_automate='"is_automation_possible" = EMPTY',
         can_be_automated='"is_automation_possible" != "false"',
         no_automation_required='"no_automation_required" = "true"'
    )),
    'disk_client': asdict(ProjectData(
         total_cases='status = "actual" AND "Тип действия" != "Я.Другое" AND "Case Type" != "CH"',
         feature_priority_key='Feature priority',
         testcase_priority_key='Test case priority',
         mark_to_automate='"is_automation_possible" = EMPTY',
         can_be_automated='"is_automation_possible" != "false"',
         no_automation_required='"no_automation_required" = "true"'
    )),
    'mobile_disk_android': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "instruction" != "true" AND "Case Type" != "CH" AND "Build Type" = "Prod"',
        feature_priority_key='feature_priority',
        testcase_priority_key='testcase_priority',
        mark_to_automate='"Automation status" = EMPTY',
        can_be_automated='"Automation status" != "Unautomatible"',
        no_automation_required='"no_automation_required" = "true"',
    )),
    'mobile_disk_ios': asdict(ProjectData(
        total_cases='status != "archived" AND status != "duplicate" AND "Case Type" != "CH" AND "Build Type" = "Prod"',
        feature_priority_key='feature_priority',
        testcase_priority_key='testcase_priority',
        mark_to_automate='"Automation status" = EMPTY',
        can_be_automated='"Automation status" != "Unautomatible"',
        no_automation_required='"no_automation_required" = "true"',
    )),
}
