from datetime import datetime
import os
from typing import Optional, Dict, List

import requests
import sys
import argparse
import yaml
from startrek_client import Startrek
import booking
from tp_api_client import tp_api_client
import logging
from data import HitmanData, StartrekData, RunData, TestpalmData

logger = logging.getLogger('testpalm_logger')
logging.basicConfig(format=u'%(asctime)s [%(levelname)s] %(module)s: %(message)s', level=logging.INFO)


def handle_options():
    parser = argparse.ArgumentParser()
    parser.add_argument('-r', '--run_process', dest="process", action='store_const', const=True, default=False,
                        help="Run only process in Hitman")
    parser.add_argument('-b', '--booking_id', dest="booking_id", help="Set booking id")
    return parser


def stop(message: str) -> None:
    logger.info(f'\x1b[31m{message}\x1b[0m')
    sys.exit(0)


def check_is_data_correct() -> None:
    data_validation = input(bold(f'Link to build: {RunData.link_to_build}\n'
                               f'Annotation to launch: {RunData.annotation}\n'
                               f'Ticket for testing: {RunData.ticket_for_testing}\n'
                               f'Is it correct? (tap \'Enter\' for continue; \'No\' for break)'))
    if data_validation is not '':
        stop('Script execution was stopped by the user')


def bold(string: str) -> str:
    return f'\x1b[1m{string}\x1b[0m'


def welcome_message():
    logger.info(f'\x1b[1m\nHi!\nNow we will create version and runs for Assessors\n'
                f'Script running with scenario {RunData.scenario}\x1b[0m')


class Booking:
    def __init__(self):
        self._id: Optional[int] = None

    @property
    def id(self) -> Optional[int]:
        return self._id

    @id.setter
    def id(self, value: int):
        self._id = value

    def book(self, testpalm_version: str) -> int:
        logger.info(f'Starting booking ...')
        book_data = booking.booking_version_testpalm(testpalm_version)
        date: str = datetime.utcfromtimestamp(int(book_data['estimate']['startTs'] / 1000)).strftime("%Y-%m-%d %H:%M:%S")
        logger.info(f'Booking created at {date}')
        self.id = book_data['bookingId']
        return self.id


class Hitman:
    def __init__(self):
        self.__oauth_token = HitmanData.token
        self.__base_url = HitmanData.base_url
        self.__headers = {
            'Authorization': f'OAuth {self.__oauth_token}',
            'Content-Type': 'application/json'
        }
        self.process_code: str = HitmanData.process_code
        self.login: str = HitmanData.login

    def run(self, testpalm_version: str, booking_id: int) -> None:
        logger.info('Creating Hitman task ...')

        if booking_id is None:
            stop('Booking id is None')

        body = {
            "requester": self.login,
            "properties": {
                "instruction": RunData.instruction,
                "namespace": TestpalmData.project,
                "special_condition": RunData.annotation,
                "parent_ticket": RunData.ticket_for_testing,
                "is_report_write_to_ticket_required": HitmanData.write_result_to_ticket,
                "test_stend": RunData.link_to_build,
                "version": testpalm_version,
                "recommendation_time": int(75),
                "tickets_queue": StartrekData.assessors_queue,
                "booking_id": booking_id
            }
        }

        response = requests.post(f"{self.__base_url}api/v1/execution/start/{self.process_code}",
                                 headers=self.__headers,
                                 json=body,
                                 verify=os.path.join(os.path.dirname(__file__), 'YandexInternalRootCA.crt'))
        response.raise_for_status()
        logger.info(f'Task in Hitman created. {response.text}')


class StarTrek:
    def __init__(self):
        self.__client = Startrek(base_url=StartrekData.base_url,
                                 useragent=StartrekData.login,
                                 token=StartrekData.token)
        self.queue: str = StartrekData.queue
        self.ticket_for_testing: str = RunData.ticket_for_testing
        self.login: str = StartrekData.login

    def add_comment_to_ticket_for_testing(self) -> None:
        self.__client.issues[self.ticket_for_testing].comments.create(text='Тестирование асессорами запущено')
        logger.info("А comment was added to the ticket for testing")


class TestPalm:
    def __init__(self, suites: List[Dict[str, str]]):
        self.client = tp_api_client.TestPalmClient(auth=TestpalmData.token)
        self.project: str = TestpalmData.project
        self.version: str = f'[{RunData.platform.value}] Assessors_{RunData.app_version.replace(".", "")}_part_{RunData.part}'
        self.suites: List[Dict[str, str]] = suites

    def create_testruns(self) -> None:
        logger.info("Creating runs ...")

        for suite in self.suites:
            run_name: str = f'[{RunData.platform.value}] {RunData.app_version.replace(".", "_")} part {RunData.part}' \
                            f' {suite["run_name"]} {suite["environment"].strip()}'
            self.create_testrun(suite=suite, title=run_name)

    def create_version(self) -> None:
        logger.info("Creating version in Testpalm ...")
        data = {'id': self.version}
        self.client.create_version(project=self.project, data=data)

    def create_testrun(self, suite: Dict[str, str], title: str) -> None:
        data = {
            'title': title,
            'version': self.version,
            'environments': [{
                'title': suite['environment'],
                'description': suite['environment'],
                'default': False
            }],
            'testSuite': {
                'id': str(suite['key'])
            },
            'tags': [suite['tags']],
            'parentIssue': {
                'assignee': [],
                'createdTime': 0,
                'display': RunData.ticket_for_testing,
                'groupId': StartrekData.queue,
                'id': RunData.ticket_for_testing,
                'isBug': False,
                'url': f'https://st.yandex-team.ru/{RunData.ticket_for_testing}',
                'status': {
                    'id': 'open',
                    'title': 'Открыт'
                },
                'isResolved': False,
                'title': StartrekData.queue,
                'trackerId': 'Startrek'
            }
        }
        self.client.create_testrun_from_suite(project=self.project, data=data)


def book_if_needed(book: Booking, book_id_arg: Optional[str], testpalm_version: str) -> None:
    if book_id_arg is not None:
        book.id = book_id_arg
    else:
        book.book(testpalm_version)


def main():
    args = handle_options().parse_args()
    suites = yaml.safe_load(open(os.path.join(os.path.dirname(__file__), 'test_suites.yaml')))
    book, hitman, testpalm, startrek = Booking(), Hitman(), TestPalm(suites[RunData.scenario]), StarTrek()

    welcome_message()
    check_is_data_correct()

    if args.process:
        book_if_needed(book, args.booking_id, testpalm.version)
        hitman.run(testpalm.version, book.id)
        stop('Script finished successfully!')

    startrek.add_comment_to_ticket_for_testing()

    testpalm.create_version()
    testpalm.create_testruns()

    book_if_needed(book, args.booking_id, testpalm.version)

    hitman.run(testpalm.version, book.id)

    logger.info(bold('Script finished successfully!'))


if __name__ == '__main__':
    main()
