import json
from pprint import pprint

import requests
from requests import Response

from pyyamail.constants import API_PREFIX, TYPE_TRASH, TYPE_INBOX
from pyyamail.tools import get_oauth_token, make_dict_filter, amp_separated, comma_separated, flatmap

class User(object):
    """
        Encapsulates API client endpoint.
        Token is enough for server communication
    """
    def __init__(self, token: str):
        self.token = token
        """:type : str"""
        self.compose_check = None
        """:type : str"""

    @staticmethod
    def create(username: str, password: str):
        token = get_oauth_token(username, password)
        return User(token)

    def execute_request(self, query: str, data=None) -> Response:
        headers = {
            'Authorization': 'OAuth ' + self.token,
            'Content-Type': 'application/json'
        }
        url = API_PREFIX + query
        print(self.__curl(url, data, headers))
        response = requests.post(url, data=data, headers=headers)
        return response

    def xlist_raw(self) -> list:
        """
            'XLIST' api query
        :return:
        """
        r = self.execute_request('xlist')
        return r.json()

    def abook_suggest_raw(self, query: str) -> dict:
        """
            'ABOOK_SUGGEST' api query

        """
        payload = {
            'query': query
        }
        r = self.execute_request('abook_suggest', payload)
        return r.json()

    def folders_raw(self) -> list:
        """
            Returns the list of folders as is
        :return:
        """
        list = self.xlist_raw()
        return [f for f in list if 'fid' in f]

    def folders(self) -> list:
        """
            Returns the list of folders with some 'useless' fields omitted
        :return:
        """
        raw = self.folders_raw()
        filt = make_dict_filter([
            'display_name',
            'type',
            'count_unread',
            'count_all',
            'fid'])
        return [filt(d) for d in raw]

    def delete(self, mids: list, fid: str) -> dict:
        """
            'DELETE_ITEMS' api query
        :param mids: mids to delete
        :param fid: current folder id
        """
        r = self.execute_request('delete_items?mids=' + comma_separated(mids) + '&current_folder=' + fid)
        return r.json()

    def messages_raw(self, payload) -> dict:
        r = self.execute_request('messages', payload)
        return r.json()

    def messages_in_folders_raw(self, fids: list, limit: int) -> dict:
        f = lambda fid: {
            "fid": fid,
            "first": 0,
            "last": limit,  # TODO what is the maximum value?
            "md5": "",
            "returnIfModified": True,
            "threaded": False
        }
        payload = {"requests": [f(fid) for fid in fids]}
        return self.messages_raw(json.dumps(payload))

    def message_header_raw(self, payload) -> dict:
        r = self.execute_request('message_header', payload)
        return r.json()

    def message_header(self, mid: int) -> dict:
        r = self.execute_request('message_header?mid=' + mid)
        return r.json()

    def messages_in_folders(self, fids: list, limit: int) -> dict:
        raw = self.messages_in_folders_raw(fids, limit)
        messages = list(flatmap(lambda d: d['messageBatch']['messages'], raw))
        filt = make_dict_filter([
            'fid',
            'firstLine',
            'mid',
            'subjPrefix',
            'subjText',
            'tid',
            'hasAttach',
            'from'])
        return [filt(d) for d in messages]

    def message_body_raw(self, mids):
        """
            'MESSAGE_BODY' api query

        """
        mids_query = ','.join(mids)
        r = self.execute_request('message_body?mids=' + mids_query)
        return r.json()

    def get_all_fids(self) -> list:
        return [f['fid'] for f in self.folders()]

    def get_all_mids(self) -> list:
        fids = self.get_all_fids()
        return [m['mid'] for m in self.messages_in_folders(fids, limit=100)]

    @staticmethod
    def find_by_type(folders: list, type: int) -> dict:
        return next(f for f in folders if f['type'] == type)

    def wipe_messages(self, completely = True) -> dict:
        """
            Erases all messages for this account
        """
        trash = User.find_by_type(self.folders(), TYPE_TRASH)

        mids = self.get_all_mids()
        if len(mids) != 0:
            self.delete(mids, trash['fid'])

        if completely:
            # First attempt might have just moved messages to trash
            return self.wipe_messages(completely= False)

    def wipe_folders(self):
        raise NotImplemented

    def wipe_labels(self):
        raise NotImplemented

    def wipe(self):
        """
            Deletes all labels, custom folders and messages
        """
        self.wipe_folders()
        self.wipe_labels()
        self.wipe_messages()

    def create_message(self,
                       fid: str = None,
                       subject: str = "",
                       content: str = "",
                       to: str = ""):
        payload = {
            "att_ids": [],
            "bcc": "",
            "cc": "",
            "compose_check": self.get_compose_check(),
            "draft_base": "",
            "send": content,
            "subj": subject,
            "to": to
        }
        if fid is not None:
            payload["store_fid"] = fid,
        r = self.execute_request("store?uuid=", json.dumps(payload))
        return r.json()

    def create_draft(self,
                     subject: str = "",
                     content: str = ""):
        return self.create_message(fid = None, subject = subject, content = content)

    def send_message(self,
                     subject: str = "",
                     content: str = "",
                     to: str = "",
                     inreplyto: str=None):
        payload = {
            "att_ids": [],
            "bcc": "",
            "cc": "",
            "compose_check": self.get_compose_check(),
            "draft_base": "",
            "send": content,
            "subj": subject,
            "to": to,
            "ttype": "html",
            "inreplyto": inreplyto
        }
        r = self.execute_request("send?uuid=", json.dumps(payload))
        return r.json()

    def create_bunch(self, fid: str, names: list):
        """
            Creates a bunch of messages in the specified folder
        :param fid: folder where the messages will be stored
        :param names: names of the messages to create (used as subject and content fields)
        """
        for name in names:
            res = self.create_message(fid, subject = name, content = name)
            pprint(res)

    def settings(self) -> dict:
        """
            SETTINGS api query
        """
        r = self.execute_request("settings")
        return r.json()

    def get_compose_check(self, force = False) -> str:
        if self.compose_check is None or force:
            s = self.settings()
            self.compose_check = s['account_information']['account-information']['compose-check']
            print(self.compose_check)
        return self.compose_check

    def setParameters(self, params: dict) -> dict:
        """
            SET_PARAMETERS api query
        :return:
        """
        payload = {
            "params": ":".join("{}={}".format(k, v) for k, v in params.items())
        }
        r = self.execute_request("set_parameters", payload)
        return r.json()

    def __curl(self, url: str, data, headers: dict):
        headers_string = ' '.join(["-H '" + name + ": " + value + "'" for name, value in headers.items()])
        return "curl '" + str(url) + "' -d '" + str(data) + "' " + str(headers_string)

#         "att_ids": [
    #     "QjtxBrGzzA4%2AVitKfZYm6LJ3iqWPwJlKZDA%2AXXFncPIM6hktBEw8bDKrNyaXqZuQMtWEvBu6P8w%3D",
    #     "RhtxBrGzzA4%2AVitKfZYm6LJ3iqWPwJlKZDA%2AXXFncPIM6hktBEw8bDKrNyaXqZuQMtWEvBu6P6e%3D"
    # ],
    # "bcc": "secret@yandex.ru",
    # "cc": "ald00@yandex.ru",
    # "compose_check": "ac5c62c4cf581815618fcb1c337b362e",
    # "draft_base": "2170000000017907704",
    # "from_mailbox": "myaddress@yandex.ua",
    # "from_name": "TEST NAME",
    # "ids": "2170000000017566230",
    # "inreplyto": "<200701381303685@web12m.yandex.ru>",
    # "parts": [
    #     "1.2",
    #     "1.3"
    # ],
    # "references": "<2504371368705087@web20d.yandex.ru> <1090901368706203@web27d.yandex.ru> <31369141970@hampers64.yandex.ru>",
    # "send": "the message body",
    # "store_fid": "2170000150000041941",
    # "subj": "the message subject",
    # "to": "devnull@ya.ru",
    # "ttype": "plain"






# curl -v --header "$AUTHORIZATION" http://mail.yandex.ru/api/mobile/v1/xlist
# wget -q -O - --header "$AUTHORIZATION" 'hattach?mid=2500000003849142688&hid=1.2&name='$NAME


