# coding=utf-8
import argparse
import json
import logging

from yql.config import config
from yql.api.v1.client import YqlClient
from datetime import datetime, timedelta

YQL_TOKEN = 'AQAD-***'

import requests

logging.basicConfig(level=logging.INFO)

IOS = 'ios'
ANDROID = 'android'

CONFIG = {
    IOS: {
        'api_key': '29733',
        'client': 'iphone'
    },
    ANDROID: {
        'api_key': '14836',
        'client': 'aphone'
    }
}


def web_url(yql_query):
    return '{}/Operations/{}'.format(config.web_url, yql_query.operation_id)


def main():
    parser = argparse.ArgumentParser(description='Find info about user pushes')
    parser.add_argument('--platform', required=True, help='Application platform', choices=[IOS, ANDROID])
    parser.add_argument('--uuids-count', required=False, help='UUID count', default=100)
    parser.add_argument('--report-json', required=True, help='Output file for json report')
    parser.add_argument('--report-html', required=True, help='Output file for html report')
    args = parser.parse_args()

    config = CONFIG[args.platform]
    api_key = config['api_key']
    uuids_count = args.uuids_count
    yesterday_date = (datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')
    query = 'USE hahn; ' \
            'PRAGMA yt.Pool = "sherlock"; ' \
            'SELECT DISTINCT `uuid`, device_id ' \
            'FROM `home/mail-logs/m3/input/daily/appmetrika/d-squeezed-log/{}` ' \
            'WHERE api_key = {} ' \
            'LIMIT {};' \
        .format(yesterday_date, api_key, uuids_count)

    request = YqlClient(token=YQL_TOKEN).query(query, syntax_version=1)
    result = request.run()
    logging.info('Started YQL: ' + web_url(request))
    result.wait_progress()
    if result.status == 'ERROR':
        raise Exception('YQL finished with error!')

    total = 0
    test_ids = {}

    for uuid, device_id in result.table.rows:
        client = config['client']
        url = 'https://mail.yandex.ru/api/mobile/v1/uaz?uuid={}&client={}'.format(uuid, client)
        logging.info('curl -X POST "{}"'.format(url))
        header = {
            'User-Agent': 'ru.yandex.mail/40270.827 (iPhone8,1; iOS 14.3)'
        }
        data = {
            'uuid': uuid,
            'device-id': device_id
        }
        try:
            response = requests.post(url, headers=header, data=data, verify=False).json()
        except:
            logging.exception('UAZ request failed!')
            continue
        status = response['status']['status']
        if status != 1:
            continue
        total += 1
        exps = list(map(lambda x: x.split(',')[0], response['exp_boxes'].split(';')))

        for test_id in exps:
            if test_id not in test_ids:
                test_ids[test_id] = 0
            test_ids[test_id] += 1

    logging.info(json.dumps(test_ids))

    stat_data = []
    for k, v in test_ids.items():
        stat_data.append({
            'application': 'mail',
            'platform': args.platform,
            'test_id': k,
            'count': v,
            'total': total,
            'percent': 100.0 * v / total,
            'fielddate': datetime.now().strftime("%Y-%m-%d")
        })
    with open(args.report_json, 'w') as f:
        f.write(json.dumps(stat_data))

    html = paint(total, test_ids)
    with open(args.report_html, 'w') as f:
        f.write(html)


def paint(total, test_ids):
    html = '<html><head><meta charset="utf-8"><title>Активные эксперименты </title></head><body>'
    html += '<table border="1" align="center"><caption>'
    html += 'Пользователей опрошено {}<br>'.format(total)
    html += '</caption>'
    html += '<tr><th>test id</th><th>У скольки пользователей</th> <th>Доля пользователей</th></tr>\n'
    for test_id, count in test_ids.items():
        html += '<tr align="center"> <td>{}</td> <td>{}</td> <td>{}%</td> </tr>\n'.format(
            test_id,
            count,
            100 * count / total
        )
    html += '</table>'
    html += '</body></html>'
    return html


if __name__ == '__main__':
    main()
