# -*- coding: utf-8 -*-
import argparse
import json
import logging
from datetime import datetime, timedelta
from yql.config import config

import requests
from yql.api.v1.client import YqlClient
import sys
reload(sys)
sys.setdefaultencoding('utf-8')

logging.basicConfig(level=logging.INFO)

IOS = 'ios'
ANDROID = 'android'

DEVELOPER_EVENTS = ['post-convey update finished', 'job update finished']

CONFIG = {
    IOS: {
        'api_key': '29733',
        'bright_push_received': 'push-received',
        'service': 'apns_queue',
        'platform': 'apns',
        'event_names': [
            'app_start',
            'push-not-permitted-by-user',
            'subscribe_for_pushes_all_scheduled',
            'subscribe_for_push_scheduled',
            'unsubscribe_from_push_scheduled',
            'subscribe_for_push',
            'unsubscribe_from_push',
            'received_new_push_token',
            'resubscribe_on_settings_update',
            'accountSwitcher_cleanup_for_deleted_account'
        ]
    },
    ANDROID: {
        'api_key': '14836',
        'bright_push_received': 'push_received',
        'service': 'mail',
        'platform': 'gcm',
        'event_names': [
            'fcm_received_new_push_token',
            'fcm_problem',
            'subscribe_for_push_scheduled',
            'subscribe_for_pushes_all_scheduled',
            'unsubscribe_from_push_scheduled',
            'subscribe_for_push',
            'unsubscribe_from_push',
            'subscribe_missing_account_id',
            'subscribe_missing_uuid',
            'manage_subscription_compat',
            'resubscribe_on_relogin_on_account_changed',
            'resubscribe_on_relogin_on_settings_update',
            'run_network_job_v2'
        ]
    }
}


DICTIONARY = {
    # xiva
    'conveyed': 'Пуш отправлен успешно',
    'convey failed': 'Ошибка при отправке пуша',
    'save failed': 'Xiva не смогла сформировать пуш',
    'xtask created': 'Нет подписки, чтобы отправить пуш',
    'dropped': 'Xiva не хочет отправлять пуш (возможно, push_token протух)',
    'subscribe finished': 'Xiva подписала на пуши',
    'unsubscribe finished': 'Xiva отписала от пушей',
    'post-convey unsubscribe finished': 'Xiva отписала от пушей из-за проблем с FCM',
    'subscribe failed': 'Xiva не удалось подписать на пуши',
    'unsubscribe failed': 'Xiva не удалось отписать от пушей',
    'post-convey callback update finished': 'Пришел новый push_token в ответе от FCM',

    # Android Mail
    'fcm_received_new_push_token': 'FCM сообщил, что надо обновить push_token',
    'fcm_problem': 'Проблема при получении push_token',
    'received_new_push_token': 'Система сообщила, что надо обновить push_token',

    'subscribe_for_pushes_all_scheduled': 'Начинаем переподписывать на пуши все аккаунты',
    'subscribe_for_push_scheduled': 'Начинаем подписываться на пуши для одного аккаунта',
    'unsubscribe_from_push_scheduled': 'Начинаем отписываться от пушей для одного аккаунта',

    'subscribe_for_push': 'Подписались на пуши',
    'unsubscribe_from_push': 'Отписались от пушей',

    'resubscribe_on_relogin_on_account_changed': 'Переподписываемся на пуши из-за изменений в аккаунте',
    'resubscribe_on_relogin_on_settings_update': 'Переподписываемся на пуши из-за смены X-Token',
    'resubscribe_on_settings_update': 'Переподписываемся на пуши из-за изменений настроек',
    'accountSwitcher_cleanup_for_deleted_account': 'Пользователь вышел из аккаунта, запустилась очистка',

    'app_start': 'Пользователь открыл приложение',
    'run_network_job_v2': 'Статистика по доступности сети'
}


class XivaSessionProvider(object):
    session_to_identifiers = {}

    def register(self, platform, uuid, device_id, client):
        session = self.__xiva_session(platform, uuid, device_id)
        self.session_to_identifiers[session] = (uuid, device_id, client)

    def session(self, platform, uuid, device_id):
        return self.__xiva_session(platform, uuid, device_id)

    def identifiers(self, session):
        return self.session_to_identifiers[session]

    def __xiva_session(self, platform, uuid, device_id):
        if platform == ANDROID:
            return uuid
        if platform == IOS:
            return device_id.lower().replace('-', '')
        raise Exception('Unknown platform ' + platform)


def appmetrika_tables(date_from, date_to):
    return range_tables('logs/metrika-mobile-log', '/30min', date_from, date_to)


def xiva_tables(corp, date_from, date_to):
    return range_tables('logs/xivahub{}-log'.format("corp" if corp else ""), '/stream/5min', date_from, date_to)

def range_tables(base_dir, today_dir, date_from, date_to):
    if not is_today(date_to):
        dir_name = base_dir + '/1d'
        return 'RANGE("{}", "{}", "{}")'.format(dir_name, date_from, date_to)
    if not is_today(date_from):
        raise Exception('"Date from" before today and "date to" today are not supported now!')

    dir_name = base_dir + today_dir
    table_name = to_date(date_from).strftime("%Y-%m-%dT00:00:00")
    return 'RANGE([{}], "{}")'.format(dir_name, table_name)


def is_today(date):
    # таблицы за предыдущий день могут появляться в течение нескольких часов
    return (datetime.now() - timedelta(hours=2)).date() <= to_date(date)


def to_date(ymd):
    return datetime.strptime(ymd, '%Y-%m-%d').date()


def to_ymd(date):
    return date.strftime('%Y-%m-%d')


def get_uid_from_env(keys_string, values_string):
    if not keys_string or not values_string:
        return None
    try:
        keys = json.loads(keys_string)
        values = json.loads(values_string)
        uid_index = keys.index('uid')
        return values[uid_index]
    except:
        return None


def get_push_data_from_bundle(value_string, event_timestamp):
    if not value_string:
        return None
    event_data = json.loads(value_string)
    bundle = event_data.get("data")
    if not bundle:
        return None, None, None

    uid = None
    transit_id = None
    operation = None
    try:
        parts = bundle.split("Bundle[{", 1)[1].split("}]", 1)[0].split(", ")
        for part in parts:
            keyvalue = part.split("=", 1)
            key = keyvalue[0]
            value = keyvalue[1] if len(keyvalue) > 1 else None
            if key == 'uid':
                uid = value
            if key == 'transit-id':
                transit_id = value
            if key == 'operation':
                operation = value
    except LookupError:
        pass

    if operation == 'insert':
        return uid, transit_id, event_timestamp
    else:
        return None, None, None


def get_push_data(platform, event_value, event_timestamp):
    if platform == IOS:
        received = json.loads(event_value)
        uid = received['uid']
        transit_id = received['transit-id']
        timestamp = datetime.fromtimestamp(float(received['timestamp'])) if 'timestamp' in received else event_timestamp
        return uid, transit_id, timestamp
    if platform == ANDROID:
        return get_push_data_from_bundle(event_value, event_timestamp)
    raise Exception('Unknown platform "{}"'.format(platform))


def analyze(platform, corp, uid, sample_uuid, xiva_token, date_from, date_to):
    if not uid and not sample_uuid:
        raise Exception('Укажите либо uid, либо UUID')

    session_provider = XivaSessionProvider()
    subscriptions = []
    if uid:
        subscriptions = find_subscriptions(platform, uid, corp, xiva_token)

    uuids = set()
    if sample_uuid:
        uuids.add(sample_uuid)
    for subscription in subscriptions:
        uuid = subscription['uuid']
        uuids.add(uuid)

    api_key = CONFIG[platform]['api_key']

    appmetrika_table = appmetrika_tables(date_from, date_to)

    qs = []
    for uuid in uuids:
        q = 'USE hahn;\n' \
            'SELECT `UUID`, DeviceID, Manufacturer, OriginalModel FROM {} WHERE APIKey = "{}" AND `UUID` = "{}" LIMIT 1;'.format(appmetrika_table, api_key, uuid)
        qs.append(q)
    tables = execute(*qs)
    device_ids = {}
    for table in tables:
        for uuid, device_id, manufacturer, original_model in table:
            client = manufacturer + ' ' + original_model
            device_ids[uuid] = device_id
            session_provider.register(platform, uuid, device_id, client)
            logging.info('Приложение UUID={} установлено на устройстве DeviceID={}'.format(uuid, device_id))

    uuids_yql = uuids_filter_yql(device_ids)
    if not uid:
        if not device_ids:
            raise Exception('Для указанного UUID={} нет логов за эти дни'.format(sample_uuid))

        q = 'USE hahn;\n' \
            'SELECT ReportEnvironment_Keys, ReportEnvironment_Values FROM {} WHERE APIKey = "{}" AND ({});'\
            .format(appmetrika_table, api_key, uuids_yql)
        uids = set()
        for keys_string, values_string in execute(q)[0]:
            uid1 = get_uid_from_env(keys_string, values_string)
            if uid1:
                uids.add(uid1)
        if len(uids) == 0:
            raise Exception('Не найдены аккаунты в этих приложениях UUID={}'.format(uuids))
        if len(uids) > 1:
            raise Exception('Найдено более одного аккаунта в этом приложении. Пожалуйста, выберите один из uid: {}'.format(uids))
        uid = list(uids)[0]
        subscriptions = find_subscriptions(platform, uid, corp)

    for subscription in subscriptions:
        if 'device_id' in subscription:
            uuid = subscription['uuid']
            device_id = subscription['device_id']
            client = subscription['client']
            device_ids[uuid] = device_id
            session_provider.register(platform, uuid, device_id, client)

    if not device_ids:
        raise Exception('Для указанных uid={} и UUID={} нет логов за эти дни'.format(uid, sample_uuid))
    uuids_yql = uuids_filter_yql(device_ids)

    event = CONFIG[platform]['bright_push_received']
    q1 = 'USE hahn;\n' \
         'SELECT `UUID`, DeviceID, EventValue, EventTimestamp, EventNumber FROM {} ' \
         'WHERE APIKey = "{}" AND ({}) AND EventName = "{}" ' \
         'ORDER BY EventTimestamp, EventNumber;'\
        .format(appmetrika_table, api_key, uuids_yql, event)

    xiva_table = xiva_tables(corp, date_from, date_to)
    session_string = ' OR '.join([
        'session = "{}"'.format(session_provider.session(platform, uuid, device_id)) for uuid, device_id in device_ids.items()
    ])
    service = CONFIG[platform]['service']
    bright_filter = 'AND event = "insert" AND uid = "{}"'.format(uid) if platform == ANDROID else ''
    q2 = 'USE hahn;\n' \
         'SELECT session, transit_id, `timestamp`, status, error, bright FROM {} ' \
         'WHERE ({}) AND group = "notifications" AND service = "{}" {} ' \
         'ORDER BY `timestamp`;'\
        .format(xiva_table, session_string, service, bright_filter)

    q3 = 'USE hahn;\n' \
         'SELECT mid, transit_id, status, error, `timestamp` FROM {} ' \
         'WHERE uid = "{}" AND service = "mail" AND event = "insert" AND session IS NULL ' \
         'ORDER BY `timestamp`;'\
        .format(xiva_table, uid)

    q4 = 'USE hahn;\n' \
         'SELECT session, status, `timestamp` FROM {} ' \
         'WHERE uid = "{}" AND service = "mail" AND group = "subscriptions" AND ({}) ' \
         'ORDER BY `timestamp`;'\
        .format(xiva_table, uid, session_string)

    event_names = CONFIG[platform]['event_names']
    event_names_yql = ' OR '.join(['EventName = "{}"'.format(e) for e in event_names])
    q5 = 'USE hahn;\n' \
         'SELECT `UUID`, DeviceID, EventName, EventValue, EventTimestamp, EventNumber FROM {} ' \
         'WHERE APIKey = "{}" AND ({}) AND ({}) ' \
         'ORDER BY EventTimestamp, EventNumber;'\
        .format(appmetrika_table, api_key, uuids_yql, event_names_yql)

    push_received_log, push_send_log, new_messages_log, xiva_subscription_log, client_subscription_log = execute(q1, q2, q3, q4, q5)

    events = []

    bright_transit_ids = set()
    silent_transit_ids = set()
    for session, transit_id, timestamp, status, error, bright in push_send_log:
        if bright == 'true':
            bright_transit_ids.add(transit_id)
            t = parse_xiva_timestamp(timestamp)
            events.append({
                'transit_id': transit_id,
                'send_time': t,
                'timestamp': t,
                'event': status,
                'error': error if error else '',
                'session': session
            })
        else:
            silent_transit_ids.add(transit_id)

    only_silent_transit_ids = silent_transit_ids - bright_transit_ids

    for mid, transit_id, status, error, timestamp in new_messages_log:
        if transit_id in only_silent_transit_ids:
            continue
        t = parse_xiva_timestamp(timestamp)
        message_event = None
        for e in events:
            if e['transit_id'] == transit_id:
                message_event = e
        if not message_event:
            message_event = {
                'transit_id': transit_id,
                'timestamp': t
            }
            events.append(message_event)
        if status == 'new':
            message_event['mid'] = mid
            message_event['message_time'] = t
        elif 'event' not in message_event:
            message_event['event'] = status
            message_event['error'] = error
            message_event['send_time'] = t

    for uuid, device_id, event_value, event_timestamp_string, event_number in push_received_log:
        event_timestamp = parse_metrica_timestamp(event_timestamp_string, event_number)
        push_uid, transit_id, timestamp = get_push_data(platform, event_value, event_timestamp)
        if push_uid != uid:
            continue

        found = False
        for e in events:
            if e['transit_id'] == transit_id:
                e['receive_time'] = timestamp
                found = True
        if not found:
            events.append({
                'transit_id': transit_id,
                'receive_time': timestamp,
                'timestamp': timestamp,
                'session': session_provider.session(platform, uuid, device_id)
            })

    for session, event, timestamp in xiva_subscription_log:
        if event in DEVELOPER_EVENTS:
            continue
        t = parse_xiva_timestamp(timestamp)
        events.append({
            'event': event,
            'timestamp': t,
            'send_time': t,
            'session': session
        })

    for uuid, device_id, event_name, event_value, event_timestamp, event_number in client_subscription_log:
        t = parse_metrica_timestamp(event_timestamp, event_number)
        events.append({
            'event': event_name,
            'error': event_value,
            'timestamp': t,
            'receive_time': t,
            'session': session_provider.session(platform, uuid, device_id)
        })

    for e in events:
        if 'session' in e:
            uuid, device_id, client = session_provider.identifiers(e['session'])
            e['uuid'] = uuid
            e['device_id'] = device_id
            e['client'] = client

    return uid, events


def uuids_filter_yql(uuids):
    return ' OR '.join(['(DeviceID = "{}" AND `UUID` = "{}")'.format(device_id, uuid) for uuid, device_id in uuids.items()])


def paint_events(uid, events):
    session_to_identifiers = {}
    for e in events:
        if 'uuid' in e and 'device_id' in e and 'session' in e and 'client' in e:
            session_to_identifiers[e['session']] = (e['uuid'], e['device_id'], e['client'])
    html = '<html><head><meta charset="utf-8"><title>Пуши пользователя {}</title></head><body>'.format(uid)
    html += '<table border="1" align="center"><caption>'
    html += 'Пуши пользователя {}<br>'.format(uid)
    for session, (uuid, device_id, client) in session_to_identifiers.items():
        html += 'Приложение {} имеет UUID={}, DeviceID={}, установлено на {}<br>'.format(session, uuid, device_id, client)
    html += '</caption>'
    html += '<tr><th>Письмо</th><th>Получено</th><th>id пуша</th> <th>Произошло в Xiva</th> <th>Событие</th> <th>Детали</th> <th>Произошло на клиенте</th> <th>Приложение (session)</th> </tr>\n'
    for e in sorted(events, key=lambda e: e['timestamp']):
        mid = e.get('mid', '')
        message_time = human_readable(e.get('message_time'))
        transit_id = e.get('transit_id', '')
        send_time = human_readable(e.get('send_time'))
        event = human_readable(e.get('event'))
        error = human_readable(e.get('error'))
        receive_time = human_readable(e.get('receive_time'))
        session = e.get('session', '')
        html += '<tr align="center"> <td>{}</td> <td>{}</td> <td>{}</td> <td>{}</td> <td>{}</td> <td>{}</td> <td>{}</td> <td>{}</td> </tr>\n'.format(
            mid,
            message_time,
            transit_id,
            send_time,
            event,
            error,
            receive_time,
            session
        )
    html += '</table>'
    html += '</body></html>'
    return html


def human_readable(s):
    if not s:
        return ''
    if s is datetime:
        return s.strftime('%Y-%m-%d %H:%M:%S')
    if s in DICTIONARY:
        return '{} <br>({})'.format(DICTIONARY[s], s)
    if not isinstance(s, basestring):
        return s
    # ХЗ, почему только в Нирване такая строка возвращается из YQL клиента
    return s.replace('\u001b[90m\u001b[0m', '').replace(',', ',<br>')


def parse_xiva_timestamp(timestamp):
    return datetime.strptime(timestamp, '%Y-%m-%d %H:%M:%S.%f')

def parse_metrica_timestamp(timestamp, event_number):
    addition_for_sort = float(event_number) / 1000000
    return datetime.fromtimestamp(float(timestamp) + addition_for_sort)


def execute(*queries):
    client = YqlClient()
    results = []
    for query in queries:
        request = client.query(query, syntax_version=1)
        result = request.run()
        logging.info('Started YQL: ' + web_url(request))
        results.append(result)

    for result in results:
        result.wait_progress()
        if result.errors:
            error_message = '; '.join(error.message for error in result.errors)
            raise Exception('YQL finished with error! Cause: ' + error_message)
        result.table.fetch_full_data()

    tables = list(map(lambda result: result.table.rows, results))
    return tables


def web_url(yql_query):
    return '{}/Operations/{}'.format(config.web_url, yql_query.operation_id)


def find_subscriptions(platform, uid, corp, xiva_token):
    service = 'mail'
    url = 'https://push.yandex{}.ru/v2/list?service={}&user={}'.format("-team" if corp else "", service, uid)
    logging.info('Получаем список активных подписок из админки Xiva: ' + url)
    resp = requests.get(url, headers={
        'Authorization': 'Xiva {}'.format(xiva_token)
    })
    resp.raise_for_status()
    matched_subscriptions = []
    for subscription in resp.json():
        if not subscription['client'].startswith('ru_yandex_mail'):
            continue
        subscription_platform = subscription['platform'] if 'platform' in subscription else 'apns'
        if CONFIG[platform]['platform'] == subscription_platform:
            uuid = subscription['session']
            subscription['uuid'] = uuid
            if platform == IOS:
                device_id = subscription['url'].split('/')[-1]
                subscription['device_id'] = device_id
            matched_subscriptions.append(subscription)
    return matched_subscriptions


def datetime_handler(x):
    if isinstance(x, datetime):
        return x.isoformat()
    raise TypeError("Unknown type")


def read_events(report_json):
    with open(report_json, 'r') as f:
        obj = json.loads(f.read())
        return obj['uid'], obj['events']


def main():
    parser = argparse.ArgumentParser(description='Find info about user pushes')
    parser.add_argument('--platform', required=True, help='Application platform', choices=[IOS, ANDROID])
    parser.add_argument('--uid', required=False, help='User uid')
    parser.add_argument('--corp', help='Add if yandex-team user', action='store_true')
    parser.add_argument('--xiva-token', required=True, help='Xiva token for list subscriptions')
    parser.add_argument('--uuid', required=False, help='Application UUID')
    parser.add_argument('--date-from', required=True, help='From date to search info')
    parser.add_argument('--date-to', required=True, help='To date to search info')
    parser.add_argument('--report-json', required=True, help='Output file for json report')
    parser.add_argument('--report-html', required=True, help='Output file for html report')

    args = parser.parse_args()
    logging.info('Started with args: ' + str(args))
    #
    # ffe780038f4323edbcda4591a141180a
    # '127f9083e92ea5012769f0c16fb5321a' '2018-12-09' '2018-12-10'
    passed_date_from = args.date_from.split('T')[0]
    passed_date_to = args.date_to.split('T')[0]

    dates_from = [passed_date_from]
    dates_to = []
    if not is_today(passed_date_from) and is_today(passed_date_to):
        dates_to.append(to_ymd(to_date(passed_date_to) - timedelta(days=1, hours=2)))
        dates_from.append(passed_date_to)
    dates_to.append(passed_date_to)

    uid = None
    all_events = []
    for date_from, date_to in zip(dates_from, dates_to):
        uid, events = analyze(args.platform, args.corp, args.uid, args.uuid, xiva_token=args.xiva_token, date_from=date_from, date_to=date_to)
        all_events.extend(events)
    # uid, events = read_events(args.report_json)

    with open(args.report_json, 'w') as f:
        f.write(json.dumps({
            'uid': uid,
            'events': all_events
        }, default=datetime_handler))

    html = paint_events(uid, all_events)
    with open(args.report_html, 'w') as f:
        f.write(html)


if __name__ == '__main__':
    main()
