# coding=utf-8
import argparse
import logging
import os
import re
import sys

from git import Repo, Git
from pyteamcity import TeamCity, GET
from startrek_client import Startrek

if sys.version_info < (3, 0):
    reload(sys)
    sys.setdefaultencoding("utf-8")

logging.basicConfig(level=logging.INFO)

class FixedTeamCity(TeamCity):
    @GET('changes/?locator=build:id:{build_id},count:1000')
    def get_changes_by_build_id(self, build_id):
        pass


def find_version(client, version_name):
    versions = client.queues['MOBILEMAIL'].versions
    for version in versions:
        if version.name == version_name:
            logging.info('Found version ' + version_name)
            return version.id
    return None


def find_issues(client, version_name):
    version_id = find_version(client, version_name)
    if not version_id:
        return []
    return client.issues.find(filter={
        'queue': 'MOBILEMAIL',
        'type': 'Release',
        'fixVersions': version_id,
    })


def last_major_version(version_name):
    parts = version_name.split('.')
    parts[-1] = '0'
    return '.'.join(parts)


def find_release_ticket(client, version_name):
    issues = find_issues(client, version_name)
    if not issues:
        major_version_name = last_major_version(version_name)
        if major_version_name != version_name:
            issues = find_issues(client, major_version_name)
    if not issues:
        raise Exception('No release ticket found by version ' + version_name)
    issue = issues[0]
    ticket = issue.key
    logging.info('Found ticket %s by version %s', ticket, version_name)
    return issue


def find_expected_tickets_in_release(client, version_name):
    version_id = find_version(client, version_name)
    if not version_id:
        return []
    issues = client.issues.find(filter={
        'queue': 'MOBILEMAIL',
        'type': 'Task',
        'fixVersions': version_id,
    })
    return sorted(map(lambda issue: issue.key, issues))


def parse_ios_version(branch_name):
    return branch_name.replace('release/', 'iOS ') + '.0' # ios release branch doesn't have .0 in name


def create_comment(version, sdk, build_id, expected_tickets, actual_tickets=None):
    expected_tickets_text = '\n'.join(map(lambda ticket: '((https://st.yandex-team.ru/' + ticket + ' ' + ticket + '))', expected_tickets))
    build_number = os.environ.get('BUILD_NUMBER', '_unknown_')
    comment = '#|\n'
    comment += "|| **Автоматический отчет о сборке билда**||\n"
    comment += "|| **Билд в Teamcity**    | ((https://teamcity.yandex-team.ru/viewLog.html?buildId=" + str(build_id) + " " + build_number + ")) ||\n"
    comment += "|| **Версия приложения**    | " + version + " ||\n"
    comment += "|| **Новые задачи в сборке** | " + expected_tickets_text + " ||\n"
    if sdk is not None:
        comment += "|| **Версии используемых SDK** | <{sdk\n" + sdk + "}> ||\n"
    comment += "|#\n"
    return comment

def parse_ticket(msg):
    _TICKET_ID_REGEXP = re.compile('[A-Z]+-[0-9]+')
    return re.findall(_TICKET_ID_REGEXP, msg)


def extract_tickets_in_release(repo):
    repo = Repo(repo)
    branch = repo.active_branch
    git = Git(repo)
    git.pull()
    branch_tickets = set()
    branch_commit_messages = git.log(branch.name, '--not', 'origin/develop', '--pretty=format:%s')
    for msg in branch_commit_messages.split():
        branch_tickets.update(parse_ticket(msg))
    return sorted(list(branch_tickets))


def read_config():
    properties_file = os.environ.get('TEAMCITY_BUILD_PROPERTIES_FILE', 'teamcity_test.properties')
    separator = "="
    config = {}
    with open(properties_file) as f:
        for line in f:
            if separator in line:
                name, value = line.split(separator, 1)
                config[name.strip()] = value.strip()
    return config


def get_teamcity_changes_safe(build_id):
    try:
        return get_teamcity_changes(build_id)
    except:
        logging.exception("Can't get changes from Teamcity!")
        return ['TEAMCITY_FAILED']


def get_teamcity_changes(build_id):
    user = os.environ['TEAMCITY_USER']
    password = os.environ['TEAMCITY_PASSWORD']
    teamcity = FixedTeamCity(user, password, 'teamcity.yandex-team.ru', 443, protocol='https')
    logging.info('Getting changes for build ' + str(build_id))
    changes = teamcity.get_changes_by_build_id(build_id)
    if 'change' not in changes:
        return []
    changes = changes['change']
    logging.info('Got %d changes', len(changes))
    if len(changes) == 1000:
        return None
    tickets = set()
    for change in changes:
        change_id = change['id']
        logging.warning('Getting message for change %s', change_id)
        teamcity_change = teamcity.get_change_by_change_id(change_id)
        comment = teamcity_change['comment']
        tickets.update(parse_ticket(comment))
    return sorted(list(tickets))


def get_android_version(path_to_build_gradle):
    with open(path_to_build_gradle, 'r') as f:
        content = f.read()
        regex = re.compile('app\s*:\s*\'([\d.]+)\'')
        versions = re.findall(regex, content)
        if versions:
            version = versions[0]
            return 'Android ' + version
    raise Exception('No "app: version" found in build.gradle')


def get_used_sdk(path_to_build_gradle):
    with open(path_to_build_gradle, 'r') as f:
        content = f.read()
        regex = re.compile('ext.versions = \[([\s\S]+?)\]') # find content of ext.versions array
        sdks = re.findall(regex, content)
        if sdks:
            sdk = re.sub('(\/\/([\s\S]+?)\n)|(\s+)', '', sdks[0]) # delete all comments, indents and \n
            sdk = re.sub(",", '\n', sdk) #replace all ',' to '\n'
            sdk = '\n'.join(sdk.split('\n')[1:]) #delete first line with app_version
            return sdk
    raise Exception('No "ext.versions" found in build.gradle')


def main():
    for k, v in os.environ.items():
        logging.info('%s: %s', k, v)

    parser = argparse.ArgumentParser(description='Great Description To Be Here')
    parser.add_argument('--repo', required=True, help='Path to git repository')
    parser.add_argument('--build-gradle', required=False, help='Path to build.gradle (Android)')

    args = parser.parse_args()

    repo = Repo(args.repo)
    active_branch_name = repo.active_branch.name
    path_to_build_gradle = args.build_gradle
    if path_to_build_gradle:
        master_names = ['master', 'mail/android/mail-app/master']
        if active_branch_name not in master_names:
            logging.info('Active branch is not master, no need to comment startrek ticket')
            return
        version = get_android_version(path_to_build_gradle)
        sdk = get_used_sdk(path_to_build_gradle)
    else:
        version = parse_ios_version(active_branch_name)
        sdk = None

    startrek_token = os.environ['STARTREK_TOKEN']
    client = Startrek(token=startrek_token, useragent='python')
    issue = find_release_ticket(client, version)

    config = read_config()
    build_id = config['teamcity.build.id']

    branch_tickets = get_teamcity_changes_safe(build_id)
    if branch_tickets is None:
        branch_tickets = find_expected_tickets_in_release(client, version)

    comment = create_comment(version, sdk, build_id, branch_tickets)
    issue.comments.create(text=comment)

    logging.info('Commented ticket %s with comment %s', issue.key, comment)


if __name__ == '__main__':
    try:
        main()
    except:
        logging.exception("Can't comment release ticket")
