import json
import os.path as path
import collections
from datetime import datetime


def normalize_phone(number):
    return number.replace('+', '').replace(' ', '').replace('-', '')


def sms_dates(sms_path):
    format = '%Y-%m-%d %H:%M:%S'
    dates = []
    with open(sms_path) as team_sms_file:
        team_sms_data = json.load(team_sms_file)
        dates = [int(rec['unixtime']) for rec in team_sms_data]
    sd = datetime.utcfromtimestamp(min(dates)).strftime(format)
    fd = datetime.utcfromtimestamp(max(dates)).strftime(format)
    return sd, fd


def sms_phones(sms_path):
    phones = collections.Counter()
    with open(sms_path) as team_sms_file:
        team_sms_data = json.load(team_sms_file)
        for number in (normalize_phone(rec['number']) for rec in team_sms_data):
            phones[number] += 1
    return phones


def staff_phones(staff_path):
    result = {}
    with open(staff_path) as staff_file:
        staff_data = json.load(staff_file)
        for person in staff_data["result"]:
            for phone in person['phones']:
                result[normalize_phone(phone['number'])] = person['login']
    return result


# Save a content of the following URL to persons.json
# https://staff-api.yandex-team.ru/v3/persons?_pretty=1&_limit=100000&_fields=login,phones&official.is_dismissed=false
# Save a content of the following URL to dismissed.json
# https://staff-api.yandex-team.ru/v3/persons?_pretty=1&_limit=100000&_fields=login,phones&official.is_dismissed=true
if __name__ == "__main__":
    data_path = path.dirname(__file__)
    team_sms_path = path.join(data_path, 'sms.yandex-team.json')
    staff_data_path = path.join(data_path, 'persons.json')
    dismissed_data_path = path.join(data_path, 'dismissed.json')

    team_sms_phones = sms_phones(team_sms_path)
    active_phones = staff_phones(staff_data_path)
    dismissed_phones = staff_phones(dismissed_data_path)
    users = [('https://staff.yandex-team.ru/' + active_phones.get(phone, dismissed_phones.get(phone, f'phone:{phone}')), n) for phone, n in team_sms_phones.most_common()]
    print(sms_dates(team_sms_path))
    print("Total SMS count: {}".format(sum((user[1] for user in users))))
    print("Total users count: {}".format(len(users)))
    print(*[f"{user}: {count}" for user, count in users], sep='\n')