import json
import re
import sys
import time
import traceback

from pyyamail.user import User

_history = {}


def dummy_tasks():
    tasks = {}
    for i in range(1, 11):
        tasks['Задача ' + str(i)] = {
            'text': 'Сколько будет 1 + ' + str(i - 1) + ' ?',
            'answer': str(i),
            'level': i
        }
    return tasks


def hardcode_tasks():
    return {
        'Задача 55': {
            'text': 'В коробке лежат 10 красны и 10 синих воздушных шариков. Продавец, не глядя, достает по одному шарику. Солько шариков надо вытащить, чтобы среди вынутых из коробки шариков обязательно нашлись 2 шарика одного цвета?',
            'answer': '3',
            'level': 2
        },
        'Задача 290': {
            'text': 'На острове живут только Лжецы и Рыцари. Лжецы всегда лгут. Рыцари всегда говорят правду. Каждый из собравшихся на острове заявил остальным <Все вы - лжецы!> Сколь рыцарей среди них?',
            'answer': '1',
            'level': 1
        },
        'Задача 314': {
            'text': 'Надпись на камне над погилой Диофанта: <Здесь погребен прах Диофанта. И числа поведать могут, о чудо, сколь долог был век его жизни. Часть шестую его представляло прекрасное детство. Двенадцатая часть протекала еще жизни - покрылся пухом тогда подбородок. Седьмую в бездетном браке провел Диофант. Прошло еще пятилетие: он был осчастливен рождением прекрасного первенца сына, коему рок половину лишь жизни прекрасной и светлой дал на земле по сравненью с отцом. И в печали глубокой старец земного удела конец воспринял, переживши года четыре с тех пор, как сына лишился>. Скольких лет жизни достигнув, смерть воспринял Диофант?',
            'answer': '84',
            'level': 5
        },
        'Задача 327': {
            'text': 'Ученик прочитал 138 страниц, что составляет 23% числа всех страниц в книге. Сколько страниц в книге?',
            'answer': '300',
            'level': 3
        },
        'Задача 363': {
            'text': 'Вот несколько айнских числительных (в латинской транскрипции): 3 - re, 11 - shine ikashma wan, 37 - arwan ikashma wan e to hotne, 47 - arwan ikashma to hotne, 93 - re ikashma wan e ashikne hotne, 135 - ashikne ikashma wan e arwan hotne. Определите, какое число записывается по-айнски как wan e re hotne.',
            'answer': '50',
            'level': 10
        },
        'Задача 876': {
            'text': 'Студенты 5 раз сдавали зачет (не сумевшие сдать зачет приходили на следующий день). Каждый раз успешно сдавала зачет треть всех пришедших студентов и еще треть студента. Каково наименьшее возможное число студентов, так и не сдавших зачет?',
            'answer': '50',
            'level': 7
        },
        'Задача 897': {
            'text': 'При делении на 2 число дает остаток 1, а при делении на 3 - остаток 2. Какой остаток дает это число при делении на 6?',
            'answer': '5',
            'level': 6
        },
        'Задача 982': {
            'text': 'Спускаясь по эскалатору, Миша наступил на 50 ступенек. А шагавший втрое быстрее БОря - на 75. Сколько ступенек на эскалаторе?',
            'answer': '100',
            'level': 9
        }
    }


def download_tasks():
    return hardcode_tasks()


all_tasks = download_tasks()


def compute_level_to_task():
    level_to_tasks = {}
    for task_key, task_info in all_tasks.items():
        level = task_info['level']
        if level not in level_to_tasks:
            level_to_tasks[level] = []
        level_to_tasks[level].append(task_key)
    return level_to_tasks


def extract_answer(body):
    print('body: ' + body)
    yandex_pattern = re.compile('>([a-zA-Z\d]+)<')
    mail_ru_pattern = re.compile('([a-zA-Z\d]+)<')
    patterns = [yandex_pattern, mail_ru_pattern]
    for pattern in patterns:
        found = pattern.findall(body)
        if found:
            return found[0]
    return


def is_correct_answer(actual_answer, real_answer):
    return actual_answer == real_answer


def get_history():
    global _history
    if not _history:
        _history = deserialize('history.json', default={})
    return _history


def store(fromm, task_title, solved):
    # TODO: dump to disk
    history_snapshot = get_history()
    if fromm not in history_snapshot:
        history_snapshot[fromm] = {}
    history_snapshot[fromm][task_title] = solved

    serialize(history_snapshot, 'history.json')


def next_task(fromm):
    level_to_tasks = compute_level_to_task()
    sorted_levels = sorted(level_to_tasks.keys())
    history_snapshot = get_history()
    if fromm not in history_snapshot:
        mid_level = sorted_levels[len(sorted_levels) // 2]
        return level_to_tasks[mid_level][0]

    min_user_level = min(sorted_levels)
    max_user_level = max(sorted_levels)
    for task, solved in history_snapshot[fromm].items():
        task_level = all_tasks[task]['level']
        if solved:
            min_user_level = max(min_user_level, task_level)
    for task, solved in history_snapshot[fromm].items():
        task_level = all_tasks[task]['level']
        if task_level >= min_user_level and not solved:
            max_user_level = min(max_user_level, task_level)
    max_user_level = max(max_user_level, min_user_level)
    optimal_level = (min_user_level + max_user_level) // 2
    for level in sorted_levels:
        if level >= optimal_level:
            for task in level_to_tasks[level]:
                solved = history_snapshot[fromm].get(task, False)
                if not solved:
                    return task
    for level in reversed(sorted_levels):
        for task in level_to_tasks[level]:
            if task not in history_snapshot[fromm]:
                return task
    return None


def bot_received(bot, message):
    fromm = message['from']['email']
    task_key = message['subjText']
    print('Received message with title ' + task_key + ' from ' + fromm)
    all_tasks = download_tasks()

    if task_key not in all_tasks:
        send_new_task(bot, fromm, message)
        return

    mid = message['mid']
    body, inreplyto = download_message_body(bot, mid)
    answer = extract_answer(body)
    print('answer: ' + answer)
    if not answer:
        return

    real_answer = all_tasks[task_key]['answer']
    solved = is_correct_answer(answer, real_answer)

    response = 'Ответ верный. Следующая задача уже в пути.' if solved else 'Неправильный ответ. Для получения другой задачи ответьте с другой темой письма.'
    store(fromm, task_key, solved)

    reply(bot, message, response)

    if solved:
        send_new_task(bot, fromm)

def download_message_body(bot, mid):
    bodies = bot.message_body_raw([mid])
    body = bodies[0]['body'][0]['content']
    inreplyto = bodies[0]['info']['references']
    return body, inreplyto

def reply(bot, incoming_message, text, subject=None):
    mid = incoming_message['mid']
    fromm = incoming_message['from']['email']
    subj = subject if subject else incoming_message['subjText']
    body, inreplyto = download_message_body(bot, mid)

    blockquote = '<blockquote style="border-left:1px solid #0857A6;margin:10px;padding:0 0 0 10px;">'
    content = text + blockquote + body + '</blockquote>'

    bot.send_message(subject=subj , content=content, to=fromm, inreplyto=inreplyto)


def send_new_task(bot, to, incoming_message=None):
    next_task_key = next_task(to)
    print('Next task key ' + next_task_key + ' for ' + to)
    if not next_task_key:
        return
    next_task_text = all_tasks[next_task_key]['text']
    if not incoming_message:
        bot.send_message(subject=next_task_key, content=next_task_text, to=to)
    else:
        reply(bot, incoming_message, next_task_text, subject=next_task_key)


def serialize(data, file):
    with open(file, 'w') as f:
        f.write(json.dumps(data))


def deserialize(file, default=None):
    try:
        with open(file, 'r') as f:
            return json.loads(f.read())
    except:
        return default


def main():
    bot = User(token='***')

    shown_mids = deserialize('shown_mids.json', default=[])

    while True:
        try:
            inbox = bot.messages_in_folders(fids=("1"), limit=1)
            for message in inbox:
                mid = message['mid']
                if mid not in shown_mids:
                    shown_mids.append(mid)
                    serialize(shown_mids, 'shown_mids.json')
                    bot_received(bot, message)
        except:
            traceback.print_exc()

        time.sleep(5)


if __name__ == '__main__':
    main()
