# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from builtins import round, all

import calendar
import sys
import datetime
import json
import math
import argparse
import businesstime
import requests
import urllib3
from urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter
import smtplib
from startrek_client import Startrek
from time import sleep
from datetime import timedelta, datetime, time
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from constants import STAT_TABLE, STAT_BASE_URL, STAT_UPLOAD, STAT_DELETE, STAT_PARAMS, STAT_TOKEN, ST_TOKEN,\
    TIME_FORMAT, QUEUES, TOTAL, PERIODS, ROBOTS, STATUS, LEADS, FILTER, SENDER_EMAIL, SENDER_PWD, HTML, SEC_IN_DAY, \
    MAX_QUALITY, MIN_QUALITY

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

client = Startrek(useragent="curl/7.53.1", token=ST_TOKEN)


def handle_options():
    parser = argparse.ArgumentParser()
    commands = parser.add_subparsers(title='sub-commands')

    upload_data_parser = commands.add_parser('upload-data')

    replace_data_parser = commands.add_parser('replace-incorrect-data')
    replace_data_parser.add_argument("--version", dest="version", required=True)
    replace_data_parser.add_argument("--queue", dest="queue", required=True)
    # replace_data_parser.add_argument("--bug_testing_time", dest="bug_testing_time")

    return parser


args = handle_options().parse_args()
action = sys.argv[1]


def connect():
    r = requests.Session()
    retries = Retry(total=5, backoff_factor=0.1, status_forcelist=[500, 502, 503, 504])
    r.mount('https://', HTTPAdapter(max_retries=retries))
    return r


def convert_to_day(ignore_night, a, b=None):
    if ignore_night:
        bt = businesstime.BusinessTime(business_hours=(time(6), time(21)))
        time_delta = bt.businesstime_hours(a, b) if b else a
    else:
        bt = businesstime.BusinessTime(business_hours=(time.min, time.max))
        time_delta = bt.businesstimedelta(a, b) if b else a
    return round(time_delta.total_seconds() / SEC_IN_DAY, 3)


def convert_to_date(string):
    return datetime.strptime(string, TIME_FORMAT)


def get_changelog(key, type):
    return list(client.issues[key].changelog.get_all(type=type))


def get_issue_data(key):
    return client.issues[key]


def get_release_fix_version(issue_data):
    try:
        fix_version = issue_data.fixVersions[0].display
    except IndexError:
        fix_version = ""
    return fix_version.replace('"', '\\"')


def get_dates_from_fields(issue_data):
    result = {}
    for key in STATUS:
        result.update({key: issue_data[key]})
    if all(_ is not None for _ in result.values()):
        for key in result:
            result[key] = convert_to_date(result[key][:19])
        return result


def get_quality(body_testing_time, regress_testing_time, release_testing_time):
    x = math.ceil((body_testing_time + regress_testing_time) * 2) / 2
    y = math.ceil(release_testing_time * 2) / 2
    quality = round(x / y * 100)
    return quality


def send_message_if_needed(queue, data):
    if data['quality'] > MAX_QUALITY or data['quality'] < MIN_QUALITY:
        recipient = LEADS[queue]
        subject, body = create_message(data)
        send_mail(subject, body, recipient)


def get_release_timings(release_ticket_data, updates_comment, updates_workflow):
    close_time = start_time = 0
    month = period = ""
    exit_close_flag = exit_start_flag = False

    # if release_ticket_data['deadline']:
    #     close_time = convert_to_date(release_ticket_data['deadline'] + "T20:59:59")

    # если релизный тикет был неправильно переведен в один из статусов
    if release_ticket_data['qaStartDate']:
        exit_start_flag = True
        start_time = convert_to_date(release_ticket_data['qaStartDate'] + "T09:00:00")

    if release_ticket_data['qaEndDate']:
        exit_close_flag = True
        close_time = convert_to_date(release_ticket_data['qaEndDate'] + "T18:59:59")

    for update in updates_workflow:
        if not exit_close_flag:
            for field in update.fields:
                if field['field'].id == 'status' and field['to'].name in ['Закрыт', 'Closed', 'Выложили', 'Released']:
                    close_time = convert_to_date(update.updatedAt[:19])
                    exit_close_flag = True
                    break
        if not exit_start_flag:
            for field in update.fields:
                if field['field'].id == 'status' and field['to'].name in ['В работе', 'In Progress', 'Тестируется', 'Testing']:
                    start_time = convert_to_date(update.updatedAt[:19])
                    exit_start_flag = True
                    break
        if exit_close_flag and exit_start_flag:
            break

    if start_time == 0:  # или первый коммент робота о пакете
        for update in updates_comment:
            if str(update.updatedBy.id) in ROBOTS:
                start_time = convert_to_date(update.updatedAt[:19])
                break

    if start_time == 0:  # и на крайний случай дату создания тикета
        start_time = convert_to_date(release_ticket_data.createdAt[:19])

    # Получаем месяц и квартал выкатки релиза
    if close_time:
        month = calendar.month_name[close_time.month]
        period = ' '.join((PERIODS[month], str(close_time.year)))

    return {
        'closeTime': close_time,
        'startTime': start_time,
        'month': month,
        'period': period
    }


def classify_tickets_in_release(fix_version, timing, fix_version_tickets):
    tickets_in_testing = []
    for ticket in fix_version_tickets:
        ticket_data = get_issue_data(ticket.key)
        ticket_type, stage, affected_version = ticket_data.type.key, ticket_data.stage, ticket_data.affectedVersions
        tags = ticket_data.tags
        created_after_start_testing = convert_to_date(ticket_data.createdAt[:19]) >= timing['startTime']

        bug_founded_in_testing_after_start = (ticket_type == 'bug' and stage == 'Testing' and created_after_start_testing)
        bug_founded_in_testing_in_current_version = (ticket_type == 'bug' and stage == 'Testing' and
                                                     ((affected_version and affected_version[0].name == fix_version) or
                                                      ('inProd' not in tags and not affected_version)))
        issue_created_after_start = (created_after_start_testing and ticket_type != 'bug')

        if bug_founded_in_testing_after_start or bug_founded_in_testing_in_current_version or issue_created_after_start:
            tickets_in_testing.append(ticket.key)

    return tickets_in_testing


def get_ticket_testing_time(tickets_in_testing):
    tickets_testing_time_in_release, tickets_testing_time_in_release_in = 0, 0
    for ticket in tickets_in_testing:
        updates_workflow = get_changelog(ticket, 'IssueWorkflow')
        start_testing_time, finish_testing_time, ticket_testing_time = 0, 0, 0
        for update in updates_workflow:
            for field in update.fields:
                if field['field'].id == 'status' and field['to'].name in ['Testing in Dev', 'Testing', 'Тестируется']:
                    start_testing_time = convert_to_date(update.updatedAt[:19])
                if field['field'].id == 'status' and field['to'].name in ['Ready for RC', 'Tested', 'Протестировано', 'Open', 'Открыт']:
                    finish_testing_time = convert_to_date(update.updatedAt[:19])

            if start_testing_time != 0 and finish_testing_time != 0 and (start_testing_time < finish_testing_time):
                ticket_testing_time_in = convert_to_day(True, start_testing_time, finish_testing_time)
                ticket_testing_time = convert_to_day(False, start_testing_time, finish_testing_time)
                tickets_testing_time_in_release_in += ticket_testing_time_in
                tickets_testing_time_in_release += ticket_testing_time
                start_testing_time, finish_testing_time = 0, 0

    return round(tickets_testing_time_in_release_in, 3), round(tickets_testing_time_in_release, 3)


def send_to_stat(queue, data, period, month, service):
    data.update({"queue": queue, "month": month, "period": period, "service": service})
    header = {'Authorization': 'OAuth ' + STAT_TOKEN}
    data_to_upload = {
        'name': STAT_TABLE,
        'scale': 'd',
        'data': json.dumps({'values': [data]})
    }

    r = connect().post(STAT_BASE_URL + STAT_UPLOAD, headers=header, data=data_to_upload)

    print(r.text)


def create_message(data):
    subject = 'Качество релиза'
    body = HTML.format(data['release'], data['quality'], data['ticket'], data['ticket'], data['body_testing_time'],
                       data['regress_testing_time'], data['release_testing_time'])
    return subject, body


def send_mail(subject, message, to):
    msg = MIMEMultipart('alternative')
    msg['Subject'] = subject
    msg['From'] = 'Release quality <{}>'.format(SENDER_EMAIL)
    msg['To'] = ','.join(to)
    msg.attach(MIMEText(message.encode('utf-8'), 'html', 'utf-8'))

    sender = SENDER_EMAIL
    sender_password = SENDER_PWD

    try:
        smtp_object = smtplib.SMTP('smtp.yandex.ru', 587)
        smtp_object.ehlo()
        smtp_object.starttls()
        smtp_object.login(sender, sender_password)
        smtp_object.sendmail(sender, to, msg.as_string())
        print('Successfully sent email to: %s' % to)
    except RuntimeError as err:
        print(err)


def delete_data_from_stat(fv):
    r = connect().post(
        STAT_BASE_URL + STAT_DELETE + STAT_TABLE + STAT_PARAMS.format(fv.replace(" ", "+")),
        headers={"Authorization": "OAuth " + STAT_TOKEN},
        json={'execute': 1}
    )

    print(r.text)


def get_release_ticket(queue, version):
    issues = client.issues.find(FILTER['RELEASE_TICKET_BY_VERSION'].format(queue, version))
    return client.issues[issues[0].key]


def upload_releases_data(release_ticket, queue):
    key = release_ticket.key
    updates_comment = get_changelog(key, 'IssueCommentAdded')
    updates_workflow = get_changelog(key, 'IssueWorkflow')
    release_ticket_data = get_issue_data(key)
    timing = get_release_timings(release_ticket_data, updates_comment, updates_workflow)
    checklist = get_dates_from_fields(release_ticket_data)

    start_testing, close_ticket = timing['startTime'], timing['closeTime']

    if checklist and close_ticket:
        finish_body_testing, finish_testing, = checklist['allTicketsTested'], checklist['versionReadyRelease'],
        start_regress, finish_regress = checklist['firstRegressionStarted'], checklist['firstRegressionDone']

        if (finish_body_testing >= start_regress) and (finish_body_testing <= finish_regress):
            body_testing_time = convert_to_day(False, start_testing, finish_regress) / 2
            regress_testing_time = convert_to_day(False, start_testing, finish_regress) / 2
            body_testing_time_in = convert_to_day(True, start_testing, finish_regress) / 2
            regress_testing_time_in = convert_to_day(True, start_testing, finish_regress) / 2
        elif (finish_body_testing >= start_regress) and (finish_body_testing > finish_regress):
            body_testing_time = convert_to_day(False, start_testing, finish_body_testing) / 2
            regress_testing_time = convert_to_day(False, start_testing, finish_body_testing) / 2
            body_testing_time_in = convert_to_day(True, start_testing, finish_body_testing) / 2
            regress_testing_time_in = convert_to_day(True, start_testing, finish_body_testing) / 2
        else:
            body_testing_time = convert_to_day(False, start_testing, finish_body_testing)
            regress_testing_time = convert_to_day(False, start_regress, finish_regress)
            body_testing_time_in = convert_to_day(True, start_testing, finish_body_testing)
            regress_testing_time_in = convert_to_day(True, start_regress, finish_regress)

        release_testing_time = convert_to_day(False, start_testing, finish_testing)
        release_testing_time_in = convert_to_day(True, start_testing, finish_testing)

        if release_testing_time > 0:
            # fix_version_tickets = []
            date = close_ticket + timedelta(hours=3)
            fix_version = get_release_fix_version(release_ticket_data)

            if not fix_version:
                fix_version = release_ticket_data.summary

            # if fix_version:
            #     fix_version_tickets = client.issues.find(FILTER['FIX_VERSION'].format(queue, fix_version, key))
            # else:
            #     fix_version = release_ticket_data.summary

            # tickets_in_testing = classify_tickets_in_release(fix_version, timing, fix_version_tickets)
            # if action == 'replace-incorrect-data' and args.bug_testing_time != 'None':
            #     bug_testing_time = float(args.bug_testing_time)
            #     bug_testing_time_in = float(args.bug_testing_time)
            # else:
            #     bug_testing_time_in, bug_testing_time = get_ticket_testing_time(tickets_in_testing)
            # quality = get_quality(body_testing_time, regress_testing_time, bug_testing_time, release_testing_time)
            # quality_in = get_quality(body_testing_time_in, regress_testing_time_in, bug_testing_time_in, release_testing_time_in)

            quality = get_quality(body_testing_time, regress_testing_time, release_testing_time)
            quality_in = get_quality(body_testing_time_in, regress_testing_time_in, release_testing_time_in)

            data = {
                'fielddate': str(date.date()),
                'date': str(date),
                'release': fix_version,
                'ticket': key,
                'release_testing_time': release_testing_time,
                'regress_testing_time': regress_testing_time,
                'body_testing_time': body_testing_time,
                'quality': quality,
                'quality_in': quality_in
            }
            print(data)

            for q in [queue, TOTAL]:
                for period in [timing['period'], TOTAL]:
                    for month in [timing['month'], TOTAL]:
                        for service in [QUEUES[queue], TOTAL]:
                            send_to_stat(q, data, period, month, service)

            send_message_if_needed(queue, data)


def main():
    if action == 'upload-data':
        for queue in QUEUES:
            release_tickets_filter = FILTER['PASSP_ISSUES'] if queue == 'PASSP' else FILTER['ALL_QUEUE_ISSUES'].format(queue)
            release_tickets = client.issues.find(release_tickets_filter)

            for release_ticket in release_tickets:
                upload_releases_data(release_ticket, queue)

    elif action == 'replace-incorrect-data':
        delete_data_from_stat(args.version)
        sleep(15)
        release_ticket = get_release_ticket(args.queue, args.version)
        upload_releases_data(release_ticket, args.queue)


if __name__ == '__main__':
    main()
