import re
import urllib
from datetime import datetime, date, timezone
from typing import Dict, List

from constants import TIME_FORMATS
import sys
import logging


def create_logger():
    logger = logging.getLogger()
    logger.setLevel(logging.INFO)
    ch = logging.StreamHandler()
    ch.setFormatter(logging.Formatter('%(asctime)s [%(levelname)s] %(module)s: %(message)s'))
    logger.addHandler(ch)
    return logger


def utc_to_local(utc_dt: datetime) -> datetime:
    return utc_dt.replace(tzinfo=timezone.utc).astimezone(tz=None)


def get_keys_with_none_value(obj: Dict, format_key_name: bool = False) -> List[str]:
    keys_with_none_value: List[str] = []
    for k, v in obj.items():
        if v is None:
            if format_key_name:
                k = convert_cammel_case_to_text(k)
            keys_with_none_value.append(k)
    return keys_with_none_value


def get_day_of_week(date: datetime = datetime.today()) -> str:
    weekdays = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday']
    weekday = date.weekday()
    return weekdays[weekday]


def convert_cammel_case_to_text(name: str) -> str:
    s1: str = re.sub('(.)([A-Z][a-z]+)', r'\1 \2', name)
    return re.sub('([a-z0-9])([A-Z])', r'\1 \2', s1).capitalize()


def convert_to_date(date: str) -> datetime:
    formatted_date = None
    for fmt in TIME_FORMATS:
        try:
            formatted_date = utc_to_local(datetime.strptime(date, fmt))
        except ValueError:
            continue
    if formatted_date is not None:
        # logging.info(f'Date {date} formatted to {formatted_date}')
        return formatted_date
    else:
        # logging.critical(f'Incorrect time format. Try one of supported: {TIME_FORMATS}')
        sys.exit()


def days_between_today_and_date(event_date: datetime) -> int:
    today = utc_to_local(datetime.today())
    print(today)
    return (today - event_date).days


def encode_url(value: str) -> str:
    return urllib.parse.quote_plus(string=value)


def ms_to_sec(ms: float) -> float:
    return ms / 1000


def unique(data: list) -> list:
    return list(set(data))


def sorted_and_unique(data: list) -> list:
    result = unique(data)
    result.sort()
    return result


def days_between_today_and_ts(ts: float) -> int:
    if ts == 0:
        return 0
    another_day = date.fromtimestamp(ms_to_sec(ts))
    today = date.today()
    return (today - another_day).days
