import smtplib
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from typing import List
from constants import MessageSender


class MailSender:
    def __init__(self,
                 recipient_list: List[str],
                 subject: str,
                 header: str,
                 body: str) -> None:
        self.__recipient_list = recipient_list
        self.__subject = subject
        self.__body = MessageSender.html_body.format(header=header, body=body)

    @property
    def subject(self) -> str:
        return self.__subject

    @subject.setter
    def subject(self, new_subject: str) -> None:
        if isinstance(new_subject, str):
            self.__subject = new_subject
        else:
            raise TypeError

    @property
    def body(self) -> str:
        return self.__body

    @body.setter
    def body(self, new_body: str) -> None:
        if isinstance(new_body, str):
            self.__body = new_body
        else:
            raise TypeError

    @property
    def recipient_list(self) -> List[str]:
        return self.__recipient_list

    @recipient_list.setter
    def recipient_list(self, new_recipient_list: List[str]) -> None:
        if isinstance(new_recipient_list, list):
            self.__recipient_list = new_recipient_list
        else:
            raise TypeError

    def create_message(self) -> str:
        msg = MIMEMultipart('alternative')
        msg['Subject'] = self.__subject
        msg['From'] = f'{MessageSender.sender_name} <{MessageSender.sender_email}>'
        msg['To'] = ','.join(self.__recipient_list)
        msg.attach(MIMEText(str(self.__body), 'html', 'utf-8'))
        return msg.as_string()

    def send(self) -> None:
        try:
            smtp_object = smtplib.SMTP(MessageSender.smpt_domain, MessageSender.smtp_port)
            smtp_object.ehlo()
            smtp_object.starttls()
            smtp_object.login(MessageSender.sender_email, MessageSender.sender_pwd)
            smtp_object.sendmail(from_addr=MessageSender.sender_email, to_addrs=self.__recipient_list, msg=self.create_message())
            print(f'Successfully sent email to: {self.__recipient_list}')
        except RuntimeError as err:
            print(err)


if __name__ == '__main__':
    message = MailSender(subject='Subject', recipient_list=['test.fanem@yandex.ru'], body=MessageSender.html_body)
    message.send()
