import json
from typing import Dict

import requests
from requests import Response
from requests.adapters import HTTPAdapter
from urllib3 import Retry
import logging


def connect():
    session = requests.Session()
    retry = Retry(total=5,
                  backoff_factor=0.3,
                  status_forcelist=(500, 502, 503, 504),
                  method_whitelist=frozenset(['GET', 'POST']))
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def log_it(method):
    def wrapper(self, *args, **kwargs):
        logging.info(f'Run function {method.__name__} with arguments {kwargs}')
        res = method(self, *args, **kwargs)
        logging.info(f'Result: {res}')
        return res
    return wrapper


class SlackSender:
    def __init__(self,
                 channel: str,
                 username: str,
                 text: str,
                 icon: str = ':white_check_mark:',
                 webhook: str = 'https://hooks.slack.com/services/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'):
        self.__channel = channel
        self.__username = username
        self.__text = text
        self.__icon = icon
        self.__webhook = webhook
        self.__headers = {'Content-Type': 'application/json'}

    def __str__(self):
        return str(self.__class__) + '\n' + '\n'.join((str(item) + ' = ' + str(self.__dict__[item]) for item in sorted(self.__dict__)))

    # Props
    @property
    def channel(self) -> str:
        return self.__channel

    @channel.setter
    def channel(self, new_channel: str) -> None:
        if isinstance(new_channel, str):
            self.__channel = new_channel
        else:
            raise TypeError

    @property
    def webhook(self) -> str:
        return self.__webhook

    @webhook.setter
    def webhook(self, new_webhook: str) -> None:
        if isinstance(new_webhook, str):
            self.__webhook = new_webhook
        else:
            raise TypeError

    @property
    def text(self) -> str:
        return self.__text

    @text.setter
    def text(self, new_text: str) -> None:
        if isinstance(new_text, str):
            self.__text = new_text
        else:
            raise TypeError

    @property
    def username(self) -> str:
        return self.__username

    @username.setter
    def username(self, new_username: str) -> None:
        if isinstance(new_username, str):
            self.__username = new_username
        else:
            raise TypeError

    @property
    def icon(self) -> str:
        return self.__icon

    @icon.setter
    def icon(self, new_icon: str) -> None:
        if isinstance(new_icon, str):
            self.__icon = new_icon
        else:
            raise TypeError

    @log_it
    def __create_message(self) -> Dict:
        return {
            "channel": self.__channel,
            "username": self.__username,
            "text": self.__text,
            "icon_emoji": self.__icon
        }

    @log_it
    def send(self) -> Response:
        response = connect().post(url=self.__webhook, headers=self.__headers, data=json.dumps(self.__create_message()))
        return response
