from tracker_api_client.tracker_client import *
from typing import Optional, List, Dict
from constants import Query, OAUTH_TOKEN_STARTREK, Comment, PR_MERGED_SUBSTRING, ROBOT_MOBDEVTOOLS, Checklist, Team, Component
from helpers import days_between_today_and_date, convert_to_date, get_keys_with_none_value, get_day_of_week
import logging

logger = logging.getLogger('tracker_logger')
logging.basicConfig(format=u'%(asctime)s [%(levelname)s] %(module)s: %(message)s', level=logging.INFO)


def log_it(method):
    def wrapper(self, *args, **kwargs):
        logging.info(f'Run function {method.__name__} with arguments {kwargs}')
        res = method(self, *args, **kwargs)
        logging.info(f'Result: {res}')
        return res

    return wrapper


class TrackerScripts:
    def __init__(self, queue: str, client) -> None:
        self.__queue = queue
        self.__client = client

    @log_it
    def create_summon_list(self, issue_id: str = None, issue: Resource = None, dev: bool = False) -> List[str]:
        author = self.__client.get_issue_author(issue_id=issue_id, issue=issue)
        components: List[str] = self.__client.get_issue_component(issue_id=issue_id, issue=issue)

        if author in Team.qa_team:
            users = [author]
        elif components is not None and Component.ios in components:
            users = Team.qa_ios.copy()
        elif components is not None and Component.android in components:
            users = Team.qa_android.copy()
        else:
            users = Team.qa_team.copy()

        if dev:
            if components is not None and Component.ios in components:
                users.extend(Team.ios_dev.copy())
            if components is not None and Component.android in components:
                users.extend(Team.android_dev_lead.copy())

        return users

    @log_it
    def close_all_tested_tickets_after_release_version(self) -> None:
        release_tickets = self.__client.find_release_tickets(queue=self.__queue, status=Status.released)
        for release_ticket in release_tickets:
            fix_versions = self.__client.get_issue_fix_version(release_ticket)
            if fix_versions is None:
                summon: List[str] = self.create_summon_list(issue_id=release_ticket)
                self.__client.add_comment(issue_id=release_ticket,
                                          text=Comment.add_release_ticket_fix_version,
                                          repeat=False,
                                          summon=summon)
                return
            for fix_version in fix_versions:
                not_closed_issues = self.__client.find_issues_with_fix_version(queue=self.__queue,
                                                                               fix_version=fix_version,
                                                                               status=Status.rc,
                                                                               type=Type.no(Type.release))
                logging.info(f'Issues: {not_closed_issues}')
                self.__client.bulk_change_issues_status(issue_ids=not_closed_issues,
                                                        transition=Transition.closed,
                                                        resolution=Resolution.fixed,
                                                        comment=Comment.change_issues_status_to_closed_due_to_release_version)

    @log_it
    def add_inprod_tag_to_backlog_bugs(self) -> None:
        query = Query.backlog_not_closed_bugs_without_tag_inprod.format(self.__queue)
        issues = self.__client.find_issues_by_query(query)
        logging.info(f'Issues: {issues}')
        self.__client.bulk_update_issues_tags(issue_ids=issues, action=Action.add, tags=[Tag.in_prod])

    @log_it
    def close_release_ticket_after_7_days(self) -> None:
        release_tickets = self.__client.find_release_tickets(queue=self.__queue, status=Status.released)
        for release_ticket in release_tickets:
            status_changed_to_released = self.__client.get_status_changes(issue_id=release_ticket, to=Status.released)
            if len(status_changed_to_released) > 0:
                date = convert_to_date(status_changed_to_released[-1]['date'])
                if days_between_today_and_date(date) >= 7:
                    logging.info(f'Release ticket {release_ticket} is closing')
                    self.__client.change_issue_status(issue_id=release_ticket,
                                                      transition=Transition.closed,
                                                      resolution=Resolution.fixed)

    @log_it
    def remove_half_year_ago_added_tag_actual(self) -> None:
        issues_with_tag_actual = self.__client.find_issues_by_query(Query.tag_actual_added_long_time_ago.format(self.__queue))
        if len(issues_with_tag_actual) > 0:
            logging.info(f'Tag Actual removing from issues {issues_with_tag_actual}')
            self.__client.bulk_update_issues_tags(issue_ids=issues_with_tag_actual, action=Action.remove, tags=[Tag.actual])
        else:
            logging.info('There are no issues')

    @log_it
    def invite_if_pr_merged_but_there_is_no_fix_version(self) -> None:
        issues = self.__client.find_issues_by_query(Query.not_closed_tickets_with_mobdevtools_robot_comments.format(self.__queue))
        issues_with_comment_about_pr_merge = []
        for issue in issues:
            comments = self.__client.get_comments(issue_id=issue,
                                                  text=PR_MERGED_SUBSTRING,
                                                  author=ROBOT_MOBDEVTOOLS)
            if len(comments) > 0:
                assignee: Optional[str] = self.__client.get_issue_assignee(issue_id=issue)
                if assignee is not None:
                    issues_with_comment_about_pr_merge.append((issue, assignee))

        for issue in issues_with_comment_about_pr_merge:
            self.__client.add_comment(issue_id=issue[0], text=Comment.add_fix_version, repeat=False, summon=[issue[1]])

    @log_it
    def invite_if_bug_weight_gt_300(self) -> None:
        issues = self.__client.find_issues_by_query(Query.tickets_with_big_weight.format(self.__queue))
        for issue_id in issues:
            summon: List[str] = self.create_summon_list(issue_id=issue_id, dev=True)
            self.__client.add_comment(issue_id=issue_id, text=Comment.fix_ticket_big_weight, repeat=False, summon=summon)

    @log_it
    def invite_if_ticket_has_fix_version_but_no_assignee(self) -> None:
        issues = self.__client.find_issues_by_query(
            Query.not_closed_tickets_with_fix_version_and_without_assignee.format(self.__queue))
        for issue_id in issues:
            summon: List[str] = self.create_summon_list(issue_id=issue_id)
            self.__client.add_comment(issue_id=issue_id, text=Comment.add_assignee, repeat=False, summon=summon)

    @log_it
    def empty_prioritization_fields(self) -> None:
        issues = self.__client.find_issues_by_query(Query.prioritization.format(self.__queue))

        for issue_id in issues:
            priority_fields = self.__client.get_issue_priority_fields(issue_id=issue_id)
            event_skip_tag = priority_fields['tags'] == ['event_skip']
            event_from_metrica_field = priority_fields['eventFromMetrica'] is not None

            if event_skip_tag and not event_from_metrica_field:
                del priority_fields['eventFromMetrica']

            keys_with_none_value: List = get_keys_with_none_value(obj=priority_fields, format_key_name=True)

            if keys_with_none_value:
                comment: str = Comment.fill_prioritization_fields.format(', '.join(keys_with_none_value))
                summon: List[str] = self.create_summon_list(issue_id=issue_id)
                self.__client.add_comment(issue_id=issue_id, text=comment, repeat=False, summon=summon)

    @log_it
    def empty_qa_fields(self) -> None:
        bug_detection_method = self.__client.find_issues_by_query(Query.bug_detection_method_empty.format(self.__queue))
        stage = self.__client.find_issues_by_query(Query.stage_empty.format(self.__queue))

        result: Dict[str, List[str]] = {}

        for issue_id in bug_detection_method:
            if issue_id not in result:
                result[issue_id] = []
            result[issue_id].append('Bug Detection Method')

        for issue_id in stage:
            if issue_id not in result:
                result[issue_id] = []
            result[issue_id].append('Stage')

        for issue_id, fields in result.items():
            comment: str = Comment.fill_fields.format(', '.join(fields))
            summon: List[str] = self.create_summon_list(issue_id=issue_id)
            self.__client.add_comment(issue_id=issue_id, text=comment, repeat=False, summon=summon)

    @log_it
    def no_new_or_old_inprod_tag(self) -> None:
        issues = self.__client.find_issues_by_query(Query.tickets_without_new_or_old_inprod.format(self.__queue))
        for issue_id in issues:
            summon: List[str] = self.create_summon_list(issue_id=issue_id)
            self.__client.add_comment(issue_id=issue_id, text=Comment.add_new_or_old_inprod_tag,
                                      repeat=False, summon=summon)

    @log_it
    def ticket_have_negative_weight(self) -> None:
        issues = self.__client.find_issues_by_query(Query.tickets_with_negative_weight.format(self.__queue))
        for issue_id in issues:
            summon: List[str] = self.create_summon_list(issue_id=issue_id)
            self.__client.add_comment(issue_id=issue_id, text=Comment.fix_ticket_negative_weight, repeat=False,
                                      summon=summon)

    @log_it
    def old_ticket_from_support(self) -> None:
        issues = self.__client.find_issues_by_query(Query.old_tickets_from_support_with_status_open.format(self.__queue))
        for issue_id in issues:
            summon: List[str] = self.create_summon_list(issue_id=issue_id)
            if Team.qa_lead[0] not in summon:
                summon.extend(Team.qa_lead)
            self.__client.add_comment(issue_id=issue_id, text=Comment.get_ticket_from_support_in_work, repeat=False,
                                      summon=summon)

    @log_it
    def closed_tickets_from_support_with_empty_qa_fields(self) -> None:
        for qa in Team.qa_team:
            issue_ids = self.__client.find_issues_by_query(
                Query.closed_tickets_from_support_with_empty_qa_fields.format(self.__queue, qa)
            )
            if len(issue_ids) > 0:
                self.__client.bulk_update_issues_field(issue_ids=issue_ids, field='qa', action=Action.add, value=qa)

    @log_it
    def empty_fields_in_release_ticket(self) -> None:
        release_tickets = self.__client.find_release_tickets(queue=self.__queue, status=Status.no(Status.closed))
        for release_ticket in release_tickets:
            release_fields = self.__client.get_issue_release_fields(issue_id=release_ticket)
            keys_with_none_value: List = get_keys_with_none_value(release_fields, format_key_name=True)
            if keys_with_none_value:
                comment: str = Comment.fill_fields.format(', '.join(keys_with_none_value))
                summon: List[str] = self.create_summon_list(issue_id=release_ticket)
                self.__client.add_comment(issue_id=release_ticket, text=comment, repeat=False, summon=summon)

    @log_it
    def inconsistency_checkbox_and_field(self) -> None:
        def checkbox_all_tickets_tested_checked(checklist: List[Dict]) -> bool:
            return len(list(filter(lambda checkbox: checkbox['text'] == Checklist.all_tickets_tested and
                                                    checkbox['checked'] is True, checklist))) > 0

        def checkbox_version_ready_release_checked(checklist: List[Dict]) -> bool:
            return len(list(filter(lambda checkbox: checkbox['text'] == Checklist.version_ready_release and
                                                    checkbox['checked'] is True, checklist))) > 0

        release_tickets = self.__client.find_release_tickets(queue=self.__queue, status=Status.no(Status.closed))
        for release_ticket in release_tickets:
            checklist = self.__client.get_issue_checklist(issue_id=release_ticket)
            is_checkbox_all_tickets_tested_checked: bool = checkbox_all_tickets_tested_checked(checklist)
            is_checkbox_version_ready_release_checked: bool = checkbox_version_ready_release_checked(checklist)

            status_changed_to_testing: bool = len(self.__client.get_status_changes(issue_id=release_ticket,
                                                                                   to=Status.testing)) > 0
            status_changed_to_tested: bool = len(self.__client.get_status_changes(issue_id=release_ticket,
                                                                                  to=Status.tested)) > 0
            status_changed_to_ready_for_release: bool = len(self.__client.get_status_changes(issue_id=release_ticket,
                                                                                             to=Status.ready_for_release)) > 0

            summon: List[str] = self.create_summon_list(issue_id=release_ticket)

            if not status_changed_to_testing and is_checkbox_all_tickets_tested_checked:
                comment = Comment.change_status_to_testing
                self.__client.add_comment(issue_id=release_ticket, text=comment, repeat=False, summon=summon)

            if not status_changed_to_tested and not status_changed_to_ready_for_release and is_checkbox_version_ready_release_checked:
                comment = Comment.change_status_to_tested
                self.__client.add_comment(issue_id=release_ticket, text=comment, repeat=False, summon=summon)


if __name__ == '__main__':
    Tracker = TrackerClient(auth=OAUTH_TOKEN_STARTREK)
    script = TrackerScripts(queue='MOBILEMAIL', client=Tracker)
    script.inconsistency_checkbox_and_field()
    script.empty_fields_in_release_ticket()
    script.close_all_tested_tickets_after_release_version()
    script.add_inprod_tag_to_backlog_bugs()
    script.close_release_ticket_after_7_days()
    # script.remove_half_year_ago_added_tag_actual()
    script.invite_if_pr_merged_but_there_is_no_fix_version()
    script.invite_if_ticket_has_fix_version_but_no_assignee()
    script.empty_prioritization_fields()
    # script.no_new_or_old_inprod_tag()
    script.ticket_have_negative_weight()
    script.old_ticket_from_support()
    script.empty_qa_fields()
    script.invite_if_bug_weight_gt_300()
    script.closed_tickets_from_support_with_empty_qa_fields()
