# -*- coding: utf-8 -*-
from __future__ import unicode_literals
import requests
import urllib3
import json
from typing import Dict, Tuple, List, Optional
from urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter
from startrek_client import Startrek
from datetime import datetime, timedelta

from constants import QUEUES, ST_TOKEN, TIME_FORMAT, STAT_BASE_URL, STAT_UPLOAD, STAT_TABLE, STAT_TOKEN, \
    DEADLINE, ROBOT_NAME, COMMENT_SUBSTRING, SLACK_ICON, SLACK_MESSAGE, SLACK_WEBHOOK, SLACK_CHANNEL, SLACK_USERNAME

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

client = Startrek(useragent="curl/7.53.1", token=ST_TOKEN)


def connect():
    session = requests.Session()
    retry = Retry(total=5,
                  backoff_factor=0.3,
                  status_forcelist=(500, 502, 503, 504),
                  method_whitelist=frozenset(['GET', 'POST']))
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def get_changelog(key: str) -> list:
    return list(client.issues[key].changelog.get_all())


def get_resolves(queue: str) -> List[str]:
    issues = client.issues.find('Queue: {} AND Tags: changed( to: finishedInYang date: today() - 1d)'.format(queue))
    # issues = client.issues.find('Queue: {} AND Tags: changed( to: finishedInYang date: 01-06-2019 .. 08-12-2019) AND Tags: changed( to: uploadedToYang date: 01-06-2019 .. 08-12-2019)'.format(queue))
    return [issue.key for issue in issues]


def convert_to_date(date: str) -> datetime:
    return datetime.strptime(date[:19], TIME_FORMAT)


def tag_in_list_starts_with(list: list, tag: str) -> bool:
    for elem in list:
        return elem.startswith(tag)


def get_start_time(issue: str, changelog: List) -> List[datetime]:
    filter_robot_comments = filter(
        lambda comment: comment["createdBy"].id == ROBOT_NAME and COMMENT_SUBSTRING in comment["text"],
        client.issues[issue].comments.get_all()
    )

    start_times = list(map(lambda comment: convert_to_date(comment["createdAt"]), filter_robot_comments))

    for update in changelog:
        updated_at = convert_to_date(update['updatedAt'])
        for field in update.fields:
            if field['field'].id == 'tags' and field['to']:
                diff = list(set(field['to']) - (set(field['from']))) if field['from'] else field['to']
                if tag_in_list_starts_with(diff, 'activeWorkers'):
                    start_times.append(updated_at)

    start_times.sort()
    return start_times


def event_between_upload_and_finish(arr: List, upload_time: datetime, finish_time: datetime) -> List[datetime]:
    return list(filter(lambda event: upload_time < event <= finish_time, arr))


def get_ticket_closing_time(changelog: List) -> List[Optional[datetime]]:
    time = []
    for update in changelog:
        updated_at = convert_to_date(update['updatedAt'])
        for field in update.fields:
            if field['field'].id == 'status' and field['to'].key == 'closed':
                time.append(updated_at)
    return time


def ticket_closed_before_finish(close: List) -> bool:
    return len(close) > 0


def comment_or_tag_exist(comments_and_tags_list):
    return len(comments_and_tags_list) > 0


def get_timings(changelog: list, start_dates: list, status: list) -> List[Dict[str, datetime]]:
    upload_time = start_time = finish_time = 0
    result = []

    def middle_between_two_dates(min_date, max_date):
        return (min_date + ((max_date - min_date) / 2)).replace(microsecond=0)

    for update in changelog:
        updated_at = convert_to_date(update['updatedAt'])

        for field in update.fields:
            if field['field'].id == 'tags' and field['to']:
                diff = list(set(field['to']) - (set(field['from']))) if field['from'] else field['to']
                # print(diff)
                if 'uploadedToYang' in diff:
                    upload_time = updated_at
                if 'finishedInYang' in diff:
                    finish_time = updated_at

                if finish_time and upload_time:
                    print('upload_time', upload_time, 'finish_time', finish_time)
                    close = event_between_upload_and_finish(arr=status,
                                                            upload_time=upload_time,
                                                            finish_time=finish_time)

                    comments_and_tags_list = event_between_upload_and_finish(arr=start_dates,
                                                                             upload_time=upload_time,
                                                                             finish_time=finish_time)

                    if ticket_closed_before_finish(close):
                        upload_time = start_time = finish_time = 0
                        continue

                    if comment_or_tag_exist(comments_and_tags_list):
                        start_time = comments_and_tags_list[0]
                    else:
                        start_time = middle_between_two_dates(min_date=upload_time, max_date=finish_time)

                    result.append({
                        'upload_time': upload_time,
                        'start_time': start_time,
                        'finish_time': finish_time
                    })

                    print('upload_time', upload_time, 'start_time', start_time, 'finish_time', finish_time)

                    upload_time = start_time = finish_time = 0

    return result


def get_duration(timings: Dict[str, datetime]) -> Tuple[int, int]:
    duration = round((timings['finish_time'] - timings['upload_time']).total_seconds() / 3600, 2)
    waiting = round((timings['start_time'] - timings['upload_time']).total_seconds() / 3600, 2)

    return duration, waiting


def met_deadline(duration: float) -> bool:
    return duration < DEADLINE


def send_to_stat(data: json) -> None:
    header = {'Authorization': 'OAuth ' + STAT_TOKEN}
    data_to_upload = {
        'name': STAT_TABLE,
        'scale': 's',
        'data': json.dumps({'values': [data]})
    }

    response = connect().post(STAT_BASE_URL + STAT_UPLOAD, headers=header, data=data_to_upload)

    print("response_status", response.status_code, "response_text", response.text)


def post_to_slack() -> None:
    webhook_url = SLACK_WEBHOOK
    data = {
        "channel": SLACK_CHANNEL,
        "username": SLACK_USERNAME,
        "text": SLACK_MESSAGE,
        "icon_emoji": SLACK_ICON
    }
    headers = {'Content-Type': 'application/json'}

    connect().post(url=webhook_url, headers=headers, data=json.dumps(data))


def main():
    for queue in QUEUES:
        resolves = get_resolves(queue)
        for resolve in resolves:
            ticket = client.issues[resolve]
            summary = ticket.summary
            key = ticket.key

            changelog = get_changelog(resolve)
            start_dates = get_start_time(issue=resolve, changelog=changelog)
            status = get_ticket_closing_time(changelog)

            timings = get_timings(changelog, start_dates, status)
            for timing in timings:
                duration, waiting = get_duration(timing)
                deadline = met_deadline(duration)

                data = {
                    'duration': duration,
                    'ticket': key,
                    'waiting': waiting,
                    'queue': queue,
                    'fielddate': str(timing['upload_time'] + timedelta(hours=3)),
                    'version': summary,
                    'deadline': deadline
                }

                print(data)
                send_to_stat(data)

    post_to_slack()


if __name__ == '__main__':
    main()
