import logging
import os
from dataclasses import dataclass
from typing import Dict

from teamcity_api_client import TeamCityClient


logger = logging.getLogger('teamcity_logger')
logging.basicConfig(format=u'%(asctime)s [%(levelname)s] %(module)s: %(message)s', level=logging.INFO)


def bytes_to_mb(b: int) -> float:
    return round(b / 1024 / 1024, 2)


@dataclass
class Constants:
    release_for_appstore_build_type = 'MobileNew_Monorepo_Mail_IOS_ReleaseForBetaAndAppStore'
    mail_pin_name = 'mail-pin'


class TeamCityScripts:
    def __init__(self, client, build_number: str):
        self.__client = client
        self.build_id: str = ''
        self.build_number: str = build_number
        self.version: str = ''
        self.pin_name: str = Constants.mail_pin_name
        self.__build_for_appstore_name: str = ''

    # Props
    @property
    def build_for_appstore_name(self) -> str:
        return self.__build_for_appstore_name

    @build_for_appstore_name.setter
    def build_for_appstore_name(self, new_build_for_appstore_name: str) -> None:
        if isinstance(new_build_for_appstore_name, str):
            self.__build_for_appstore_name = new_build_for_appstore_name
        else:
            raise TypeError

    def get_build_info_by_build_number(self) -> Dict[str, str]:
        build_list = self.__client.get_build_list({
            'buildType': Constants.release_for_appstore_build_type,
            'defaultFilter': 'false',
            'status': 'SUCCESS',
            # 'pinned': 'false'
        })

        if build_list['count'] == 0:
            raise ValueError('There are no builds')

        build = list(filter(lambda b: b['number'] == self.build_number, build_list['build']))

        if len(build) == 0:
            raise ValueError(f'There is no build with build number {self.build_number}')

        self.build_id, self.version = build[0]['id'], build[0]['branchName']
        self.build_for_appstore_name = f'YandexMobileMail-{self.version}-{self.build_number}-for-AppStore.ipa'

        return {
            'id': self.build_id,
            'build_number': self.build_number,
            'version': self.version
        }

    def write_appstore_build_size_to_wiki(self):
        artifact_metadata = self.__client.get_information_about_build_artifact(
            f'id:{self.build_id}',
            [self.build_for_appstore_name]
        )
        return bytes_to_mb(int(artifact_metadata['size']))

    def pin_release_build_and_add_tag(self):
        self.__client.pin_build(f'id:{self.build_id}')
        self.__client.add_tags(f'id:{self.build_id}', [self.pin_name])


if __name__ == '__main__':
    teamcity_scripts = TeamCityScripts(TeamCityClient(auth="AQAD-***"),
                                       build_number='800')
    # size = teamcity_scripts.write_appstore_build_size_to_wiki()
    # print(size)
    print(teamcity_scripts.get_build_info_by_build_number())
    print(teamcity_scripts.write_appstore_build_size_to_wiki())
