import json
import logging
import os
import re
from furl import furl
from dataclasses import dataclass
from typing import Dict, Optional, List
import requests
from requests import Response
from requests.adapters import HTTPAdapter
from urllib3 import Retry

logger = logging.getLogger('teamcity_logger')


@dataclass
class Constants:
    base_url = 'https://teamcity.yandex-team.ru'
    url_path_prefix = ['app', 'rest']


def log_it(method):
    def wrapper(self, *args, **kwargs):
        logger.debug(f'Run function {method.__name__} with arguments {kwargs}')
        res = method(self, *args, **kwargs)
        logger.debug(f'Result: {res}')
        return res

    return wrapper


class TeamCityClient:
    def __init__(self, auth, host=Constants.base_url):
        self.__auth = auth
        self.__host = host
        self.__headers = {
            'Authorization': f'OAuth {self.__auth}',
            'Content-Type': 'application/json',
            'Accept': 'application/json, text/*'
        }

    @property
    def host(self) -> str:
        return self.__host

    @host.setter
    def host(self, new_host: str) -> None:
        self.__host = new_host

    @staticmethod
    def __connect():
        r = requests.Session()
        retries = Retry(total=5,
                        backoff_factor=0.3,
                        status_forcelist=[500, 502, 503, 504],
                        method_whitelist=frozenset(['GET', 'POST']))
        adapter = HTTPAdapter(max_retries=retries)
        r.mount('https://', adapter)
        r.mount('http://', adapter)
        r.verify = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'cacert.pem')
        return r

    @staticmethod
    def __serialize_to_json(data: Dict) -> Optional[bytes]:
        if data is None:
            return data
        return json.dumps(data, ensure_ascii=False, separators=(',', ': ')).encode('utf-8')

    def __update_headers_if_needed(self, headers: Optional[Dict[str, str]] = None) -> Dict[str, str]:
        new_headers = self.__headers.copy()
        if headers:
            for header, value in headers.items():
                new_headers[header] = value
        return new_headers

    @staticmethod
    def __format_build_locator(build_locator: Dict[str, str]) -> str:
        return re.sub("[{} ']", '', str(build_locator))

    @log_it
    def __make_get_request(self, url: str) -> Response:
        response = self.__connect().get(url, headers=self.__headers)
        logger.debug(response.text)
        print(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_post_request(self, url: str, data=None, files=None) -> Response:
        if files is not None:
            del self.__headers["Content-Type"]
        response = self.__connect().post(url, headers=self.__headers, data=self.__serialize_to_json(data), files=files)
        logger.debug(response.text)
        print(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_delete_request(self, url: str, data=None, headers: Optional[Dict[str, str]] = None) -> Response:
        headers = self.__update_headers_if_needed(headers)
        response = self.__connect().delete(url, data=self.__serialize_to_json(data), headers=headers)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_put_request(self, url: str, data, headers: Optional[Dict[str, str]] = None) -> Response:
        headers = self.__update_headers_if_needed(headers)
        response = self.__connect().put(url, headers=headers, data=data)
        logger.debug(response.text)
        print(response.text)
        response.raise_for_status()
        return response

    @log_it
    def get_information_about_build_artifact(self, build_locator: str, path_to_artifact: List[str]) -> json:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'artifacts', 'metadata'] + path_to_artifact)
        response = self.__make_get_request(url)
        return response.json()

    # Tags
    @log_it
    def get_tags(self, build_locator: str) -> Dict:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'tags'])
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def add_tags(self, build_locator: str, tags: List[str]) -> str:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'tags'])
        data = {'tag': [{'name': tag} for tag in tags]}
        response = self.__make_post_request(url, data=data)
        return response.text

    @log_it
    def replace_tags(self, build_locator: str, tags: List[str]) -> str:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'tags'])
        data = {'tag': [{'name': tag} for tag in tags]}
        response = self.__make_put_request(url, data=data)
        return response.text

    @log_it
    def delete_tags(self, build_locator: str, tags: List[str]) -> Optional[str]:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'tags'])
        current_tags = self.get_tags(build_locator)
        if current_tags['count'] == 0:
            logger.warning('There are no tags')
            return
        result_tags = {'tag': [tag for tag in current_tags['tag'] if tag['name'] not in tags]}
        response = self.__make_put_request(url, data=result_tags)
        return response.text

    # Pin
    @log_it
    def get_pin_status(self, build_locator: str) -> bool:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'pin'])
        response = self.__make_get_request(url)
        return bool(response.text)

    @log_it
    def pin_build(self, build_locator: str, message: str = '') -> None:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'pin'])
        self.__make_put_request(url, data=message, headers={'Content-Type': 'text/plain'})
        logger.info(f'Build {build_locator} pinned with message {message}')

    @log_it
    def unpin_build(self, build_locator: str, message: str = '') -> None:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds', build_locator, 'pin'])
        self.__make_delete_request(url, data=message, headers={'Content-Type': 'text/plain'})
        logger.info(f'Build {build_locator} unpinned with message {message}')

    # Builds
    @log_it
    def get_build_list(self, build_locator: Dict[str, str]) -> json:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + ['builds']) \
            .add(query_params={'locator': self.__format_build_locator(build_locator)})
        response = self.__make_get_request(url)
        return response.json()
