import json
import requests
from dataclasses import dataclass
from requests import Response
from urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter
from typing import Any, Dict, List, Optional
from furl import furl
import logging
import os

OLD_API_HOST = 'https://testpalm.yandex-team.ru'
NEW_API_HOST = 'https://testpalm-api.yandex-team.ru'

logger = logging.getLogger('testpalm_logger')


def log_it(method):
    def wrapper(self, *args, **kwargs):
        logger.debug(f'Run function {method.__name__} with arguments {kwargs}')
        res = method(self, *args, **kwargs)
        logger.debug(f'Result: {res}')
        return res

    return wrapper


@dataclass
class Projects:
    mobmail_ios = 'mobmail_ios'
    mobmail_android = 'mobmail_android'
    mobilemail = 'mobilemail'


@dataclass
class RunStatus:
    created = 'CREATED'
    started = 'STARTED'
    finished = 'FINISHED'


@dataclass
class VersionStatus:
    created = 'CREATED'
    started = 'STARTED'
    finished = 'FINISHED'


@dataclass
class CaseStatus:
    actual = 'actual'
    draft = 'draft'
    on_review = 'on review'
    needs_changes = 'needs changes'
    automated = 'automated'
    automation_in_progress = 'automation in progress'
    needs_repair = 'needs repair'
    duplicate = 'duplicate'
    archived = 'archived'


@dataclass
class Fields:
    version = 'version'
    status = 'status'
    suites = 'suites'
    started_time = 'startedTime'
    created_by = 'createdBy'


@dataclass
class Operators:
    eq = "EQ"
    neq = "NEQ"
    gt = "GT"
    lt = "LT"
    include = "IN"
    not_include = "NIN"


@dataclass
class Action:
    add = 'add'
    remove = 'remove'
    replace = 'replace'
    empty = 'empty'


@dataclass
class LinkType:
    parent = 'parent'
    child = 'child'
    related = 'related'


class TestPalmClient:
    def __init__(self, auth, host=NEW_API_HOST):
        self.__auth = auth
        self.__host = host
        self.__headers = {
            'Authorization': f'OAuth {self.__auth}',
            'Content-Type': 'application/json'
        }

    @property
    def host(self) -> str:
        return self.__host

    @host.setter
    def host(self, new_host: str) -> None:
        self.__host = new_host

    @staticmethod
    def __raise_if_include_with_exclude_param(kwargs) -> None:
        if 'exclude' in kwargs and 'include' in kwargs:
            raise ValueError(f"'Include' param can't be used with 'exclude' param.")

    @staticmethod
    def __compare_actual_and_expected_args(expected_args: List[str], actual_args: List[str]) -> None:
        for arg in actual_args:
            if arg not in expected_args:
                logger.warning(f'Unexpected argument {arg}. Expected args: {expected_args}')

    @staticmethod
    def __connect():
        r = requests.Session()
        retries = Retry(total=5,
                        backoff_factor=0.3,
                        status_forcelist=[500, 502, 503, 504],
                        method_whitelist=frozenset(['GET', 'POST']))
        adapter = HTTPAdapter(max_retries=retries)
        r.mount('https://', adapter)
        r.mount('http://', adapter)
        r.verify = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'cacert.pem')
        return r

    @staticmethod
    def __serialize_to_json(data: Dict) -> Optional[bytes]:
        if data is None:
            return data
        return json.dumps(data, ensure_ascii=False, separators=(',', ': ')).encode('utf-8')

    @log_it
    def __make_get_request(self, url: str) -> Response:
        response = self.__connect().get(url, headers=self.__headers)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_post_request(self, url: str, data=None, files=None) -> Response:
        if files is not None:
            del self.__headers["Content-Type"]
        response = self.__connect().post(url, headers=self.__headers, data=self.__serialize_to_json(data), files=files)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_patch_request(self, url: str, data) -> Response:
        response = self.__connect().patch(url, headers=self.__headers, data=self.__serialize_to_json(data))
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_delete_request(self, url: str, data=None) -> Response:
        response = self.__connect().delete(url, data=self.__serialize_to_json(data), headers=self.__headers)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_put_request(self, url: str, data) -> Response:
        response = self.__connect().put(url, headers=self.__headers, data=self.__serialize_to_json(data))
        logger.debug(response.text)
        response.raise_for_status()
        return response

    # Project
    @log_it
    def get_project(self, project: str, **kwargs) -> Dict:
        url = furl(self.__host) \
            .add(path=['projects', project])\
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_project(self, data: Dict[str, Any]) -> Dict:
        url = furl(self.__host) \
            .add(path=['projects'])
        response = self.__make_post_request(url, data=data)
        return response.json()

    @log_it
    def update_project(self, project: str, data: Dict[str, Any]) -> Dict:
        url = furl(self.__host) \
            .add(path=['projects', project])
        response = self.__make_put_request(url, data=data)
        logger.info(f'Project {project} updated successfully.')
        return response.json()

    @log_it
    def clone_project(self, project: str, new_project: str, title: str = "Copy",
                      clone_testcases: bool = True, clone_testruns: bool = True, clone_versions: bool = True) -> Dict:
        url = furl(self.__host) \
            .add(path=['projects', project, 'clone', new_project]) \
            .add(args={
                        'title': title,
                        'cloneTestcases': clone_testcases,
                        'cloneTestRuns': clone_testruns,
                        'cloneVersions': clone_versions
                       })
        response = self.__make_post_request(url, data={})
        logger.info(f'Project {project} successfully cloned to {new_project} project.')
        return response.json()

    @log_it
    def delete_project(self, project: str, permanent: bool = False) -> None:
        url = furl(self.__host) \
            .add(path=['projects', project])\
            .add(args={"permanent": permanent})
        self.__make_delete_request(url)
        logger.info(f'Project {project} deleted successfully.')

    # Versions
    @log_it
    def get_versions(self, project: str, statuses: List[str] = None, title: str = None, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['version', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        result = response.json()

        if statuses:
            result = list(filter(lambda x: x["status"] in statuses, result))
        if title:
            result = list(filter(lambda x: x["title"] == title, result))

        return result

    @log_it
    def get_version(self, project: str, version_id: str, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['version', project, version_id]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_version(self, project: str, data: Dict[str, Any]) -> None:
        already_created = len(self.get_versions(project, title=data['id'], include='title')) > 0
        if already_created:
            logger.info(f'Version with title "{data["id"]}" already created')
            return None
        else:
            url = furl(self.__host) \
                .add(path=['version', project])
            self.__make_post_request(url, data=data)
            logger.info(f'Version with title "{data["id"]}" was created successfully')

    @log_it
    def update_version(self, project: str, data: Dict[str, Any]) -> List:
        url = furl(self.__host) \
            .add(path=['version', project])
        response = self.__make_patch_request(url, data=data)
        logger.info(f'Version with title "{data["id"]}" was updated successfully')
        return response.json()

    @log_it
    def add_comment_for_version(self, project: str, version_id: str, comment: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['version', project, version_id, 'comments'])
        response = self.__make_post_request(url, data={"text": comment})
        logger.info(f'Comment "{comment}" for version {version_id} was added successfully')
        return response.json()

    @log_it
    def update_version_comment(self, project: str, comment_id: str, comment: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['version', project, 'comments', comment_id])
        response = self.__make_patch_request(url, data={"text": comment})
        logger.info(f'Comment "{comment}" was updated successfully')
        return response.json()

    # Testruns
    @log_it
    def get_testruns(self, project: str, **kwargs) -> List[Dict]:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        self.__compare_actual_and_expected_args(expected_args=['expression', 'limit', 'include', 'exclude', 'skip'],
                                                actual_args=list(kwargs.keys()))
        url = furl(self.__host) \
            .add(path=['testrun', project])\
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testruns_by_status(self, project: str, **kwargs) -> Dict[str, List[Dict[str, int]]]:
        self.__raise_if_include_with_exclude_param(kwargs.keys())

        url = furl(self.__host) \
            .add(path=['testrun', project, 'by_status']) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testrun(self, project: str, testrun_id: str, **kwargs) -> List[Dict]:
        self.__raise_if_include_with_exclude_param(kwargs.keys())

        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testcase_from_testrun(self, project: str, testrun_id: str, case_id: int, **kwargs) -> List[Dict]:
        self.__raise_if_include_with_exclude_param(kwargs.keys())

        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, case_id]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_testrun_from_suite(self, project: str, data: Dict[str, Any]) -> None:
        url = furl(self.__host) \
            .add(path=['testrun', project, 'create'])
        self.__make_post_request(url, data=data)
        logger.info(f'Testrun {data["title"]} was created successfully')

    @log_it
    def create_testrun_from_cases(self, project: str, data: Dict[str, Any]) -> None:
        url = furl(self.__host) \
            .add(path=['testrun', project, 'create']) \
            .add({'includeOnlyExisted': True})
        self.__make_post_request(url, data=data)
        logger.info(f'Testrun {data["title"]} was created successfully')

    @log_it
    def update_testrun(self, project: str, data: Dict[str, Any]) -> None:
        url = furl(self.__host) \
            .add(path=['testrun', project])
        self.__make_patch_request(url, data=data)
        logger.info(f'Testrun {data["title"]} was updated successfully')

    @log_it
    def get_testcase_uuid_in_testrun(self, project: str, testrun_id: str, case_id: int):
        testrun = self.get_testrun(project=project, testrun_id=testrun_id)
        testcases_groups = list(map(lambda testGroup: testGroup['testCases'], testrun['testGroups']))
        testcases = [item for sublist in testcases_groups for item in sublist]

        testcases_with_id = list(filter(lambda testcase: testcase['testCase']['id'] == case_id, testcases))

        if len(testcases_with_id) > 0:
            uuids = list(map(lambda testcase: testcase['uuid'], testcases_with_id))

            if len(uuids) > 1:
                raise ValueError(
                    f'There are {len(uuids)} {uuids} testcases  with id {case_id} in testrun {testrun_id}.'
                    f'Run function with one of uuid')
        else:
            raise ValueError(f'There is no testcase with id {case_id} in testrun {testrun_id}')

        return uuids[0]

    @log_it
    def finish_testrun(self, project: str, testrun_id: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, 'finish'])
        response = self.__make_post_request(url)
        return response.json()

    @log_it
    def get_testrun_comments(self, project: str, testrun_id: str, **kwargs) -> List:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, 'comments'])\
            .add(args=kwargs)
        response = self.__make_get_request(url)
        logger.info(f'Testrun {testrun_id} was finished successfully')
        return response.json()

    @log_it
    def add_testrun_comment(self, project: str, testrun_id: str, comment: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, 'comments'])
        response = self.__make_post_request(url, data={'text': comment})
        logger.info(f'Comment for run {testrun_id} was added successfully')
        return response.json()

    @log_it
    def update_testrun_comment(self, project: str, testrun_id: str, comment_id: str, comment: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, 'comments', comment_id])
        response = self.__make_patch_request(url, data={'text': comment})
        logger.info(f'Comment for run {testrun_id} was updated successfully')
        return response.json()

    @log_it
    def delete_testrun_comment(self, project: str, testrun_id: str, comment_id: str) -> None:
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, 'comments', comment_id])
        self.__make_delete_request(url)
        logger.info(f'Comment {comment_id} for run {testrun_id} was deleted successfully')

    @log_it
    def add_comment_for_testcase_in_testrun(self, project: str, testrun_id: str, comment: str,
                                            case_id: int = None, case_uuid: str = None) -> Dict:
        if case_uuid is None and case_id is not None:
            case_uuid = self.get_testcase_uuid_in_testrun(project, testrun_id, case_id)
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, case_uuid, 'comments'])
        response = self.__make_post_request(url, data={'text': comment})
        logger.info(f'Comment for case {case_uuid} was added successfully')
        return response.json()

    @log_it
    def get_testcase_comments_in_testrun(self, project: str, testrun_id: str,
                                         case_id: int = None, case_uuid: str = None, **kwargs) -> List:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        if case_uuid is None and case_id is not None:
            case_uuid = self.get_testcase_uuid_in_testrun(project, testrun_id, case_id)
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, case_uuid, 'comments'])\
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def update_testcase_comment_in_testrun(self, project: str, testrun_id: str, comment_id: str, new_comment: str,
                                           case_id: int = None, case_uuid: str = None) -> List:
        if case_uuid is None and case_id is not None:
            case_uuid = self.get_testcase_uuid_in_testrun(project, testrun_id, case_id)
        url = furl(self.__host) \
            .add(path=['testrun', project, testrun_id, case_uuid, 'comments'])
        response = self.__make_patch_request(url, data={'id': comment_id, 'text': new_comment})
        return response.json()

    # Testcases
    @log_it
    def get_testcase(self, project: str, case_id: int, **kwargs) -> List:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        url = furl(self.__host) \
            .add(path=['testcases', project]) \
            .add(args={'id': case_id}) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testcases(self, project: str, **kwargs) -> List:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        self.__compare_actual_and_expected_args(expected_args=['expression', 'limit', 'include',
                                                               'exclude', 'skip', 'searchQuery'],
                                                actual_args=list(kwargs.keys()))
        url = furl(self.__host) \
            .add(path=['testcases', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def delete_testcases(self, project: str, case_ids: List[int], permanently: bool = False) -> None:
        path = ['testcases', project]
        type = 'permanent' if permanently else 'bulk'
        path.append(type)
        url = furl(self.__host) \
            .add(path=path)
        self.__make_delete_request(url, data=case_ids)
        logger.info(f'Testcases with ids {case_ids} was deleted successfully.')

    @log_it
    def create_testcase(self, project: str, data: Dict[str, Any]) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', project])
        self.__make_post_request(url, data=data)
        logger.info('Testcase was created successfully')

    @log_it
    def create_testcases(self, project: str, data: List[Dict[str, Any]]) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', project, 'bulk'])
        self.__make_post_request(url, data=data)
        logger.info('Testcase was created successfully')

    @log_it
    def update_testcase(self, project: str, data: Dict[str, Any]) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', project])
        self.__make_patch_request(url, data=data)
        logger.info('Testcase was updated successfully')

    @log_it
    def update_testcases(self, project: str, data: List[Dict[str, Any]]) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', project, 'bulk'])
        self.__make_patch_request(url, data=data)
        logger.info('Testcase was updated successfully')

    def create_attribute_value_if_needed(self, project: str, attr_name: str, attr_values: List[str] = []) -> None:
        existed_attr_values = self.get_project_definition(project=project, title=attr_name)[0]["values"]
        for new_attr_value in attr_values:
            if new_attr_value not in existed_attr_values:
                logger.info(f'There is no "{attr_values}" value in "{attr_name}" attribute in {project} project. '
                            f'New value will be added.')
                self.update_project_definition_values(project=project, action=Action.add,
                                                      title=attr_name, values=[new_attr_value])

    def create_attribute_if_needed(self, project: str, attr_name: str, attr_values: List[str]) -> Dict[str, str]:
        id_by_title = self.get_attributes_id_by_title(project)
        if attr_name not in id_by_title:
            logger.info(f'There is no "{attr_name}" attribute in {project} project. New attribute will be created.')
            self.create_project_definition(project=project, title=attr_name, values=attr_values)
            id_by_title = self.get_attributes_id_by_title(project)
        return id_by_title

    def __execute_action_on_testcase_attribute(self, cases: List[Dict], attribute_id: str,
                                               attr_values: List[str], action: str) -> List[Dict]:
        if action == Action.replace:
            for case in cases:
                case['attributes'][attribute_id] = attr_values
        elif action == Action.add:
            for case in cases:
                case['attributes'][attribute_id].extend(attr_values)
        elif action == Action.remove:
            for case in cases:
                for value in attr_values:
                    if value in case['attributes'][attribute_id]:
                        case['attributes'][attribute_id].remove(value)
                    else:
                        raise ValueError(f'There is no "{value}" in {case["attributes"][attribute_id]}')
        elif action == Action.empty:
            for case in cases:
                case['attributes'][attribute_id] = []
        else:
            raise ValueError('Uknown action. Use one of: replace, add, remove, empty')
        return cases

    @log_it
    def update_testcases_attribute_value(self, project: str, case_ids: List[int], action: str,
                                         attr_name: str, attr_values: List[str] = []) -> None:
        if len(case_ids) < 1:
            logger.info('case_ids list is empty')
            return

        id_by_title = self.create_attribute_if_needed(project=project, attr_name=attr_name, attr_values=attr_values)
        self.create_attribute_value_if_needed(project=project, attr_name=attr_name, attr_values=attr_values)
        attribute_id = id_by_title[attr_name]

        def split_list(lst, n):
            return [lst[i:i + n] for i in range(0, len(lst), n)]

        list_of_cases_lists = []
        for case_ids in split_list(case_ids, 100):
            list_of_cases_lists.append(self.get_testcases(project=project,
                                                          include='id,attributes,createdBy',
                                                          expression=str({"type": Operators.include,
                                                                          "key": "id",
                                                                          "value": case_ids}).replace("'", '"')))
        cases = [case for cases_list in list_of_cases_lists for case in cases_list]
        for case in cases:
            case['createdBy'] = 'robot-mobmail-qa' if 'createdBy' not in case else case['createdBy']

            if attribute_id not in case['attributes']:
                logger.info(f"Case doesn't has attribute with title '{attr_name}'")
                case['attributes'][attribute_id] = []

        cases = self.__execute_action_on_testcase_attribute(cases=cases, attribute_id=attribute_id,
                                                            attr_values=attr_values, action=action)
        self.update_testcases(project=project, data=cases)
        logger.info(f'Testcases attribute {attr_name} was {action} successfully. New value - {attr_values}')

    @log_it
    def get_testcases_from_suite(self, project: str, suite_id: str, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['testcases', project, 'suite', suite_id])\
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testcases_events_by_project(self, project: str, **kwargs) -> List:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        url = furl(self.__host) \
            .add(path=['eventslog', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testcase_links(self, project: str, case_id: int) -> List:
        url = furl(self.__host) \
            .add(path=['testcases', 'link', project, case_id])
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_testcase_link(self, project: str, case_id: int, link_type: str,
                            linking_project: str, linking_case_id: int) -> List:
        data = {
          "projectId": linking_project,
          "testcaseId": linking_case_id,
          "type": link_type
        }
        url = furl(self.__host) \
            .add(path=['testcases', 'link', project, case_id])
        response = self.__make_post_request(url, data=data)
        return response.json()

    @log_it
    def delete_testcase_link(self, project: str, case_id: int, link_id: str) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', 'link', project, case_id, link_id])
        self.__make_delete_request(url)
        logger.info(f'Link {link_id} with case {project}-{case_id} was removed successfully')

    @log_it
    def delete_tracker_link(self, project: str, case_id: int, st_issue: str, tracker_id: str = 'Startrek') -> None:
        url = furl(OLD_API_HOST) \
            .add(path=['api', 'tracker', tracker_id, 'backlink', project, 'testcase', case_id, 'issue', st_issue])
        self.__make_delete_request(url)
        logger.info(f'Link {case_id} with {st_issue} was removed successfully')

    @log_it
    def add_comment_for_testcase(self, project: str, case_id: int, comment: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['testcases', project, case_id, 'comments'])
        response = self.__make_post_request(url, data={"text": comment})
        return response.json()

    @log_it
    def get_testcase_comments(self, project: str, case_id: int) -> List[Dict]:
        url = furl(self.__host) \
            .add(path=['testcases', project, case_id, 'comments'])
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def delete_testcase_comment(self, project: str, comment_id: str) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', project, 'comments', comment_id])
        self.__make_delete_request(url)
        logger.info(f'Comment with id {comment_id} was removed successfully')

    @log_it
    def add_testcase_attachment(self, project: str, case_id: int, path_to_attachment: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['testcases', project, case_id, 'attachment'])
        files = {'upload_file': open(path_to_attachment, 'rb')}
        response = self.__make_post_request(url, files=files)
        return response.json()

    @log_it
    def get_testcase_attachments(self, project: str, case_id: int) -> Dict:
        attachments = self.get_testcase(project=project, case_id=case_id, include='attachments')[0]['attachments']
        return attachments

    @log_it
    def save_to_file_testcase_attachment(self, project: str, case_id: int,
                                         attachment_id: str, path_to_dir: str = './') -> None:
        attachments = self.get_testcase_attachments(project=project, case_id=case_id)
        attachment_name = list(map(lambda attachment: attachment['name'],
                                   filter(lambda attachment: attachment['id'] == attachment_id, attachments)))[0]
        url = furl(self.__host) \
            .add(path=['testcases', project, case_id, 'attachment', attachment_id])
        response = self.__make_get_request(url)
        with open(f'{path_to_dir}{attachment_name}', 'wb') as f:
            f.write(response.content)
        logger.info(f'Attachment {attachment_name} was saved to file {path_to_dir}{attachment_name} successfully')

    @log_it
    def delete_testcase_attachment(self, project: str, case_id: int, attachment_id: str) -> None:
        url = furl(self.__host) \
            .add(path=['testcases', project, case_id, 'attachment', attachment_id])
        self.__make_delete_request(url)
        logger.info(f'Attachment {attachment_id} was deleted successfully')

    def clone_testcases(self, source_project: str, destination_project: str, case_ids: List[int]) -> Dict:
        url = furl(self.__host) \
            .add(path=['testcases', "clone", source_project, destination_project])
        response = self.__make_post_request(url, data=case_ids)
        new_case_ids = list(map(lambda case: case["id"], response.json()))
        logger.info(f'Testcases {case_ids} from project {source_project} to project {destination_project} '
                    f'successfully cloned with ids {new_case_ids}')
        return response.json()

    # Definitions
    @log_it
    def get_project_definitions(self, project: str, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['definition', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_project_definition(self, project: str, title: str) -> List:
        url = furl(self.__host) \
            .add(path=['definition', project])\
            .add(args={"expression": str({"type": Operators.eq, "key": "title", "value": title}).replace("'", '"')})
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_project_definition(self, project: str, title: str, values: List[str]) -> None:
        definitions = self.get_project_definitions(project)
        definition_already_created = len(list(filter(lambda definition: definition["title"] == title, definitions))) > 0
        if definition_already_created:
            logger.info(f'Definition {title} already created')
            return
        url = furl(self.__host) \
            .add(path=['definition', project])

        data = {
            "title": title,
            "removed": False,
            "hidden": False,
            "restricted": False,
            "values": values
        }
        self.__make_put_request(url, data)
        logger.info(f'Project definition {title} was created successfully')

    @log_it
    def update_project_definition_values(self, project: str, action: str, title: str, values: List[str]) -> None:
        url = furl(self.__host) \
            .add(path=['definition', project])
        definitions = self.get_project_definitions(project)
        definition = list(filter(lambda definition: definition["title"] == title, definitions))[0]

        if action == Action.replace:
            definition['values'] = values
        elif action == Action.add:
            definition['values'].extend(values)
        elif action == Action.remove:
            for value in values:
                if value in definition['values']:
                    definition['values'].remove(value)
                else:
                    raise ValueError(f'There is no "{value}" in {definition["values"]}')
        elif action == Action.empty:
            definition['values'] = []
        else:
            raise ValueError('Uknown action. Use one of: replace, add, remove, empty')
        self.__make_put_request(url, definition)
        logger.info(f'Project definition {title} was {action} successfully')

    @log_it
    def get_attributes_id_by_title(self, project: str) -> Dict[str, str]:
        definitions = self.get_project_definitions(project)
        id_by_title = {}
        for definition in definitions:
            id_by_title.update({definition["title"]: definition["id"]})
        return id_by_title

    # Testsuites
    @log_it
    def get_testsuites(self, project: str, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['testsuite', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testsuite(self, project: str, testsuite_id: str, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['testsuite', project, testsuite_id]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_testsuite(self, project: str, data: Dict[str, Any]) -> List:
        url = furl(self.__host) \
            .add(path=['testsuite', project])
        response = self.__make_post_request(url, data=data)
        return response.json()

    @log_it
    def update_testsuite(self, project: str, data: Dict[str, Any]) -> List:
        url = furl(self.__host) \
            .add(path=['testsuite', project])
        response = self.__make_patch_request(url, data=data)
        return response.json()

    @log_it
    def update_testsuites(self, project: str, data: List[Dict[str, Any]]) -> List:
        url = furl(self.__host) \
            .add(path=['testsuite', project, 'bulk'])
        response = self.__make_patch_request(url, data=data)
        return response.json()

    # Testplan
    @log_it
    def get_testplans(self, project: str, **kwargs) -> List:
        url = furl(self.__host) \
            .add(path=['testplan', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_testplan(self, project: str, testplan_id: str, **kwargs) -> Dict:
        url = furl(self.__host) \
            .add(path=['testplan', project, testplan_id]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_testplan(self, project: str, data: Dict[str, Any]) -> Dict[str, Any]:
        url = furl(self.__host) \
            .add(path=['testplan', project])
        response = self.__make_post_request(url, data=data)
        title = response.json()['title']
        logger.info(f'Testplan {title} was created successfully')
        return response.json()

    @log_it
    def update_testplan(self, project: str, data: Dict[str, Any]) -> Dict[str, Any]:
        url = furl(self.__host) \
            .add(path=['testplan', project])
        response = self.__make_patch_request(url, data=data)
        title = response.json()['title']
        logger.info(f'Testplan {title} was updated successfully')
        return response.json()

    @log_it
    def delete_testplan(self, project: str, testplan_id: str) -> None:
        url = furl(self.__host) \
            .add(path=['testplan', project, testplan_id])
        self.__make_delete_request(url)
        logger.info(f'Testplan was deleted successfully')

    # Shared step
    @log_it
    def get_shared_steps(self, project: str, **kwargs) -> List[Dict]:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        url = furl(self.__host) \
            .add(path=['step', project]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_shared_step(self, project: str, step_id: str, **kwargs) -> Dict:
        self.__raise_if_include_with_exclude_param(kwargs.keys())
        url = furl(self.__host) \
            .add(path=['step', project]) \
            .add(args={'id': step_id}) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()[0]

    @log_it
    def create_shared_step(self, project: str, **kwargs) -> Dict:
        self.__compare_actual_and_expected_args(expected_args=['step', 'expect', 'tags'],
                                                actual_args=list(kwargs.keys()))
        url = furl(self.__host) \
            .add(path=['step', project])
        response = self.__make_post_request(url, data=kwargs)
        logger.info(f'Shared step was created successfully')
        return response.json()

    @log_it
    def update_shared_step(self, project: str, data: Dict[str, Any]) -> Dict:
        url = furl(self.__host) \
            .add(path=['step', project])
        response = self.__make_patch_request(url, data=data)
        logger.info(f'Shared step was updated successfully')
        return response.json()

    @log_it
    def delete_shared_step(self, project: str, step_id: str) -> Dict:
        url = furl(self.__host) \
            .add(path=['step', project, step_id])
        response = self.__make_delete_request(url)
        logger.info(f'Shared step was deleted successfully')
        return response.json()

    # Organization
    @log_it
    def get_organization(self, organization_id: str, **kwargs) -> Dict:
        url = furl(self.__host) \
            .add(path=['organizations', organization_id]) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_organization(self, data: Dict) -> Dict:
        url = furl(self.__host) \
            .add(path=['organizations'])
        response = self.__make_post_request(url, data=data)
        return response.json()

    @log_it
    def update_organization(self, organization_id: str, data: Dict) -> Dict:
        url = furl(self.__host) \
            .add(path=['organizations', organization_id])
        response = self.__make_patch_request(url, data=data)
        return response.json()

    # Automation group
    @log_it
    def get_automation_group(self, project: str, automation_id: str) -> List[Dict]:
        url = furl(self.__host) \
            .add(path=['automationgroup', project, automation_id])
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def create_automation_group(self, project: str, data: Dict[str, Any]) -> List[Dict]:
        url = furl(self.__host) \
            .add(path=['automationgroup', project])
        response = self.__make_post_request(url, data=data)
        return response.json()

    # Order
    @log_it
    def set_order(self, project: str, testcase_ids: List[int]) -> List[Dict]:
        url = furl(self.__host) \
            .add(path=['order', project])
        response = self.__make_post_request(url, data=testcase_ids)
        return response.json()

    # Audit
    @log_it
    def get_definition_audit(self, project: str, **kwargs) -> List[Dict]:
        url = furl(self.__host) \
            .add(path=['audit', project, 'definition'])\
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()

    # Runner
    @log_it
    def get_runner_config_format(self, **kwargs) -> List[Dict]:
        url = furl(self.__host) \
            .add(path=['runner', 'all']) \
            .add(args=kwargs)
        response = self.__make_get_request(url)
        return response.json()
