import logging
from dataclasses import dataclass
from typing import Optional, List, Dict, Any
from startrek_client import Startrek
from startrek_client.objects import Resource
from datetime import datetime, timezone

logger = logging.getLogger('tracker_logger')

PRIORITIZATION_FIELDS = ['tags', 'audience', 'bugType', 'crash', 'eventFromMetrica', 'howReproduce']


class ClassWithChangeMethod:
    @classmethod
    def no(cls, value: str) -> str:
        return f"! {value}"

    @classmethod
    def changed(cls,
                frm: Optional[str] = None,
                to: Optional[str] = None,
                by: Optional[str] = None,
                date: Optional[str] = None) -> Optional[str]:
        result = []
        if frm:
            result.append(f'from: {frm}')
        if to:
            result.append(f'to: {to}')
        if by:
            result.append(f'by: {by}')
        if date:
            result.append(f'date: {date}')

        if frm is None and to is None and by is None and date is None:
            raise ValueError
        else:
            return f"changed({' '.join(result)})"


@dataclass(frozen=True)
class Date:
    today = 'today()'
    yesterday = 'yesterday()'
    current_week = 'week()'
    current_month = 'month()'
    current_quarter = 'quarter()'
    half_year = '>= today() - 6month'
    year = '>= today() - 1year'
    more_half_year_ago = '<= today() - 6month'


@dataclass(frozen=True)
class Priority:
    blocker = 'blocker'
    critical = 'critical'
    minor = 'minor'
    normal = 'normal'
    trivial = 'trivial'


@dataclass
class Type(ClassWithChangeMethod):
    bug = 'bug'
    task = 'task'
    release = 'release'
    technical_task = 'technicalTask'


@dataclass(frozen=True)
class Transition:
    reopen = 'reopen'
    close = 'close'
    closed = 'closed'
    need_info = 'needInfo'
    testing = 'testing'
    tested = 'tested'
    release = 'release'
    ready_for_release = 'readyForRelease'
    rc = 'rc'
    testing_in_dev = 'tesingInDev'
    tested_in_dev = 'testedInDev'
    merged = 'merged'
    ready_for_test = 'readyForTest'
    ready_for_development = 'readyForDevelopment'


@dataclass
class Tag(ClassWithChangeMethod):
    actual = 'Actual'
    new_in_prod = 'newInProd'
    old_in_prod = 'oldInProd'
    in_prod = 'inProd'
    resolve_qa = 'Resolve_QA'


@dataclass(frozen=True)
class Resolution:
    fixed = "fixed"
    wont_fix = "won'tFix"
    cant_reproduce = "can'tReproduce"
    duplicate = "duplicate"
    invalid = "invalid"
    later = "later"
    empty = None


@dataclass
class Status(ClassWithChangeMethod):
    open = 'open'
    closed = 'closed'
    need_info = 'needInfo'
    testing = 'testing'
    tested = 'tested'
    ready_for_release = 'readyForRelease'
    released = 'released'
    rc = 'rc'
    testing_in_dev = 'tesingInDev'
    tested_in_dev = 'testedInDev'
    merged = 'merged'
    ready_for_test = 'readyForTest'
    ready_for_development = 'readyForDevelopment'


@dataclass
class Relationships:
    relates = 'relates'                       # обычная связь
    is_dependent_by = 'is dependent by'       # текущий тикет является блокером линкуемого
    depends_on = 'depends on'                 # текущий тикет зависит от линкуемого
    is_subtask_for = 'is subtask for'         # текущий тикет является сабтаском линкуемого
    is_parent_task_for = 'is parent task for' # текущий тикет является родительским таском линкуемого
    duplicates = 'duplicates'                 # текущий тикет дублирует линкуемый
    is_duplicated_by = 'is duplicated by'     # текущий тикет дублирован линкуемым
    is_epic_of = 'is epic of'                 # текущий тикет является эпиком линкуемого
    has_epic = 'has epic'                     # текущий тикет является подэпиком линкуемого
    original = 'original'                     # текущий тикет является оригиналом линкуемого
    clone = 'clone'                           # текущий тикет является клоном линкуемого


@dataclass
class RemoteServices:
    arcanum = 'ru.yandex.arcanum'
    brak = 'ru.yandex.brak'
    conductor = 'ru.yandex.conductor'
    crucible = 'ru.yandex.crucible'
    github = 'com.github'
    lunapark = 'ru.yandex.lunapark'
    otrs = 'ru.yandex.otrs'
    procu = 'ru.yandex.procu'
    reviewboard = 'org.reviewboard'
    sandbox = 'ru.yandex.sandbox'
    stash = 'ru.yandex.stash'
    testpalm = 'ru.yandex.testpalm'
    zendesk = 'com.zendesk'


@dataclass
class Fields:
    event_from_metrica = 'eventFromMetrica'
    how_reproduce = 'howReproduce'
    testing_story_points = 'testingStoryPoints'
    bug_detection_method = 'bugDetectionMethod'
    audience = 'audience'
    reproducibility = 'reproducibility'
    crash = 'crash'
    bug_type = 'bugType'
    feature = 'feature'
    weight_one = 'weightOne'
    number_of_complaints = 'numberOfComplaints'
    stage = 'stage'


@dataclass
class Action:
    add = 'add'
    remove = 'remove'
    set = 'set'
    unset = 'unset'


TIME_FORMATS = ('%Y-%m-%dT%H:%M:%S.%f', '%Y-%m-%d %H:%M:%S.%f', '%Y-%m-%dT%H:%M:%S.%f%z',
                '%Y-%m-%d %H:%M:%S.%f%z', '%Y-%m-%dT%H:%M:%S', '%Y-%m-%d %H:%M:%S', '%Y-%m-%d')


def utc_to_local(utc_dt: datetime) -> datetime:
    return utc_dt.replace(tzinfo=timezone.utc).astimezone(tz=None)


def convert_to_date(date: str) -> datetime:
    formatted_date = None
    for fmt in TIME_FORMATS:
        try:
            formatted_date = utc_to_local(datetime.strptime(date, fmt))
        except ValueError:
            continue
    if formatted_date is not None:
        return formatted_date
    else:
        raise ValueError('Converting to date failed. Unknown format of time.')


def testpalm_case(project: str, case_id: str) -> str:
    return f'testcases/{project}/{case_id}'


def testpalm_run(project: str, run_id: str) -> str:
    return f'testrun/{project}/{run_id}'


def log_it(method):
    def wrapper(self, *args, **kwargs):
        logger.info(f'Run function {method.__name__} with arguments {kwargs}')
        res = method(self, *args, **kwargs)
        if res is not None:
            logger.info(f'Result: {res}')
        return res

    return wrapper


class TrackerClient:
    def __init__(self, auth: str) -> None:
        self.__auth = auth
        self.__client = Startrek(useragent="curl/7.53.1", token=self.__auth)

    # Getting values from issue fields
    @log_it
    def get_issue(self, issue_id: str):
        return self.__client.issues[issue_id]

    @log_it
    def get_issue_checklist(self, issue_id: str = None, issue: Resource = None) -> List[str]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.checklistItems:
            return list(issue.checklistItems)
        else:
            raise ValueError('Issue has no checklist')

    @log_it
    def get_issue_fix_version(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.fixVersions:
            fix_versions = list(map(lambda fix_version: fix_version.display, issue.fixVersions))
            return fix_versions

    @log_it
    def get_issue_affected_version(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.affectedVersions:
            affected_versions = list(map(lambda affected_version: affected_version.display, issue.affectedVersions))
            return affected_versions

    @log_it
    def get_issue_assignee(self, issue_id: str = None, issue: Resource = None) -> Optional[str]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.assignee:
            return issue.assignee.id
        else:
            return None

    @log_it
    def get_issue_author(self, issue_id: str = None, issue: Resource = None) -> Optional[str]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.createdBy:
            return issue.createdBy.id

    @log_it
    def get_issue_status(self, issue_id: str = None, issue: Resource = None) -> Optional[str]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.status:
            return issue.status.key

    @log_it
    def get_issue_followers(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.followers:
            return list(map(lambda follower: follower.id, issue.followers))

    @log_it
    def get_issue_tags(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.tags:
            return issue.tags

    @log_it
    def get_issue_sprint(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.sprint:
            return list(map(lambda sprint: sprint.display, issue.sprint))

    @log_it
    def get_issue_type(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.type:
            return list(map(lambda type: type.key, issue.type))

    @log_it
    def get_issue_component(self, issue_id: str = None, issue: Resource = None) -> Optional[List[str]]:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.components:
            return list(map(lambda component: component.display, issue.components))

    @log_it
    def get_issue_priority_fields(self, issue_id: str = None, issue: Resource = None) -> Dict[str, Any]:
        fields: List[str] = PRIORITIZATION_FIELDS
        result: Dict[str, Any] = dict.fromkeys(fields, None)
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None:
            for field in fields:
                if field == 'tags' and issue[field]:
                    priority_tags = list(filter(lambda tag: tag in ['event_skip', 'event_add'], issue[field]))
                    if priority_tags:
                        result['tags'] = priority_tags
                elif issue[field]:
                    result[field] = issue[field]
        return result

    @log_it
    def get_issue_release_fields(self, issue_id: str = None, issue: Resource = None) -> Dict[str, Any]:
        fields: List[str] = ['releaseNotes', 'releaseType', 'deadline', 'qa']
        result: Dict[str, Any] = dict.fromkeys(fields, None)
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None:
            for field in fields:
                if field == 'qa' and issue[field]:
                    qa = list(map(lambda qa: qa.id, issue[field]))
                    if qa:
                        result['qa'] = qa
                elif issue[field]:
                    result[field] = issue[field]
        return result

    @log_it
    def get_issue_weight(self, issue_id: str = None, issue: Resource = None) -> int:
        if issue_id and not issue:
            issue = self.get_issue(issue_id)
        if issue is not None and issue.weightOne:
            return issue.weightOne

    # Create|update issue
    @log_it
    def create_issue(self, **params) -> Resource:
        issue = self.__client.issues.create(**params)
        logger.info(f'Issue {issue["key"]} was created successfully')
        return issue

    # Move issue
    @log_it
    def move_issue(self, issue_id: str, queue: str) -> Resource:
        new_issue = self.__client.issues[issue_id].move_to(queue=queue)
        return new_issue

    # Update issue
    @log_it
    def update_issue(self, issue_id: str, **params) -> Resource:
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Issue was updated successfully')
        return issue

    @log_it
    def update_issue_field(self, issue_id: str, field: str, action: str, value) -> Resource:
        params = {field: {action: value}}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Field {field} was updated successfully')
        return issue

    @log_it
    def update_issue_assignee(self, issue_id: str, action: str, value) -> Resource:
        params = {"assignee": {action: value}}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Assignee was updated successfully')
        return issue

    @log_it
    def update_issue_type(self, issue_id: str, type: str) -> Resource:
        params = {"type": type}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Issue type was updated successfully')
        return issue

    @log_it
    def update_issue_priority(self, issue_id: str, priority: str) -> Resource:
        params = {"priority": priority}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Issue priority was updated successfully')
        return issue

    @log_it
    def update_issue_tags(self, issue_id: str, action: str, tags: List[str]) -> Resource:
        params = {"tags": {action: tags}}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Tags {tags} was updated successfully')
        return issue

    @log_it
    def update_issue_fix_version(self, issue_id: str, action: str,
                                 version_ids: List[int] = None, version_names: List[str] = None) -> Resource:
        if version_ids is None and version_names is not None:
            queue_versions = self.get_queue_versions(queue=issue_id.split('-')[0])
            version_ids = []
            for name in version_names:
                version_ids.append(list(filter(lambda version: version['name'] == name, queue_versions))[0].id)
        params = {"fixVersions": {action: version_ids}}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Fix version {version_ids} was updated successfully')
        return issue

    @log_it
    def update_issue_affected_version(self, issue_id: str, action: str, version_ids: List[int] = None,
                                      version_names: List[str] = None) -> Resource:
        if version_ids is None and version_names is not None:
            queue_versions = self.get_queue_versions(queue=issue_id.split('-')[0])
            version_ids = []
            for name in version_names:
                version_ids.append(list(filter(lambda version: version['name'] == name, queue_versions))[0].id)
        params = {"affectedVersions": {action: version_ids}}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Affected version {version_ids} was updated successfully')
        return issue

    @log_it
    def update_issue_component(self, issue_id: str, action: str, component_ids: List[int] = None,
                               component_names: List[str] = None) -> Resource:
        if component_ids is None and component_names is not None:
            queue_components = self.get_queue_components(queue=issue_id.split('-')[0])
            component_ids = []
            for name in component_names:
                component_ids.append(
                    list(filter(lambda component: component['name'] == name, queue_components))[0].id)
        params = {"components": {action: component_ids}}
        issue = self.__client.issues[issue_id].update(**params)
        logger.info(f'Component {component_ids} was updated successfully')
        return issue

    @log_it
    def change_issue_status(self, issue_id: str, transition: str, resolution: Optional[str] = None) -> List[Resource]:
        return self.__client.issues[issue_id].transitions[transition].execute(resolution=resolution)

    # Comments
    @log_it
    def get_comments(self, issue_id: str, text: str = None, author: str = None,
                     from_date_utc: str = None, to_date_utc: str = None) -> List[Dict[str, Any]]:
        comments = list(map(lambda comment: {
                                        "text": comment.text,
                                        "date": comment.createdAt,
                                        "author": comment.createdBy.id,
                                        "id": comment.id
                                    }, self.__client.issues[issue_id].comments.get_all()))
        if text:
            comments = list(filter(lambda comment: text in comment["text"], comments))
        if author:
            comments = list(filter(lambda comment: comment["author"] == author, comments))
        if from_date_utc:
            comments = list(filter(lambda comment: convert_to_date(comment["date"]) > convert_to_date(from_date_utc), comments))
        if to_date_utc:
            comments = list(filter(lambda comment: convert_to_date(comment["date"]) < convert_to_date(to_date_utc), comments))
        return comments

    @log_it
    def get_comment(self, issue_id: str, comment_id: int) -> List[Dict[str, Any]]:
        return self.__client.issues[issue_id].comments[comment_id]

    @log_it
    def add_comment(self, issue_id: str, text: str, summon: List[str] = None,
                    repeat: bool = True, attachments: List[str] = None) -> None:
        if not repeat and len(self.get_comments(issue_id=issue_id, text=text)) > 0:
            logger.info(f'Comment {text} was added earlier')
            return
        self.__client.issues[issue_id].comments.create(text=text, summonees=summon, attachments=attachments)
        logger.info(f'Comment to issue {issue_id} with text "{text}" was added successfully')

    @log_it
    def delete_comment(self, issue_id: str, comment_id: int) -> None:
        self.__client.issues[issue_id].comments[comment_id].delete()
        logger.info(f'Comment with id {comment_id} was deleted successfully')

    @log_it
    def update_comment(self, issue_id: str, comment_id: int, text: str, attachments: List[str] = None) -> None:
        self.__client.issues[issue_id].comments[comment_id].update(text=text, attachments=attachments)
        logger.info(f'Comment with id {comment_id} was updated successfully')

    # Issue links
    @log_it
    def get_links(self, issue_id: str) -> List[Dict[str, Any]]:
        links = list(map(lambda link: {
            "type": link.type.id,
            "direction": link.direction,
            "issue_key": link.object.key,
            "issue_name": link.object.display,
            "author": link.createdBy.id,
            "created_at": link.createdAt,
            "updated_at": link.updatedAt,
            "id": link.id
        }, self.__client.issues[issue_id].links.get_all()))
        return links

    @log_it
    def get_link(self, issue_id: str, link_id: int) -> Dict[str, Any]:
        link = self.__client.issues[issue_id].links[link_id]
        return {
            "type": link.type.id,
            "direction": link.direction,
            "issue_key": link.object.key,
            "issue_name": link.object.display,
            "author": link.createdBy.id,
            "created_at": link.createdAt,
            "updated_at": link.updatedAt,
            "id": link.id
        }

    @log_it
    def add_link(self, issue_id: str, relationship: str, to: str) -> None:
        self.__client.issues[issue_id].links.create(relationship, to)
        logger.info(f'Link with type {relationship} with issue {to} added successfully')

    @log_it
    def delete_link(self, issue_id: str, linked_issue_id: str = None, link_id: int = None) -> None:
        if linked_issue_id is not None and link_id is None:
            links = self.get_links(issue_id=issue_id)
            link_id = list(map(lambda link: link['id'], links))[0]
        self.__client.issues[issue_id].links[link_id].delete()
        logger.info(f'Link with id {link_id} deleted successfully')

    # Issue remote links
    @log_it
    def get_remote_links(self, issue_id: str):
        remote_links = list(map(lambda link: {
            "type": link.type.id,
            "direction": link.direction,
            "issue_key": link.object.key,
            "remote_service_type": link.object.application.type,
            "remote_service_name": link.object.application.name,
            "author": link.createdBy.id,
            "created_at": link.createdAt,
            "updated_at": link.updatedAt,
            "id": link.id
        }, self.__client.issues[issue_id].remotelinks.get_all()))
        return remote_links

    @log_it
    def get_remote_link(self, issue_id: str, link_id: int) -> Dict[str, Any]:
        link = self.__client.issues[issue_id].remotelinks[link_id]
        return {
            "type": link.type.id,
            "direction": link.direction,
            "issue_key": link.object.key,
            "remote_service_type": link.object.application.type,
            "remote_service_name": link.object.application.name,
            "author": link.createdBy.id,
            "created_at": link.createdAt,
            "updated_at": link.updatedAt,
            "id": link.id
        }

    @log_it
    def add_remote_link(self, issue_id: str, origin: str, key: str, relationship: str = Relationships.relates) -> None:
        self.__client.issues[issue_id].remotelinks.create(origin=origin, key=key, relationship=relationship)
        logger.info(f'Remote link with type {relationship} with {origin.split(".")[-1]} [issue {key}] added successfully')

    @log_it
    def delete_remote_link(self, issue_id: str, linked_remote_issue_id: str = None, remote_link_id: int = None) -> None:
        if linked_remote_issue_id is not None and remote_link_id is None:
            remote_links = self.get_remote_links(issue_id=issue_id)
            remote_link_id = list(map(lambda link: link['id'], remote_links))[0]
        self.__client.issues[issue_id].remotelinks[remote_link_id].delete()
        logger.info(f'Remote link with id {remote_link_id} deleted successfully')

    # Finding issues
    @log_it
    def find_issues_by_query(self, query: str, per_page: int = None) -> List[str]:
        issues = self.__client.issues.find(query=query, per_page=per_page)
        return [issue.key for issue in issues]

    @log_it
    def find_issues_by_filter(self, filter: Dict[str, str] = None, filter_id: int = None, order: str = None) -> List[str]:
        issues = self.__client.issues.find(filter=filter, filter_id=filter_id, order=order)
        return [issue.key for issue in issues]

    @log_it
    def find_release_tickets(self, queue: str, status: str) -> List[str]:
        issues = self.__client.issues.find(f'Queue: {queue} AND Status: {status} AND Type: {Type.release}')
        return [issue.key for issue in issues]

    @log_it
    def find_issues_with_fix_version(self, queue: str, fix_version: str,
                                     status: Optional[str] = None, type: Optional[str] = None) -> List[str]:
        query = f'Queue: {queue} AND "Fix Version": "{fix_version}"'
        if status:
            query += f" AND Status: {status}"
        if type:
            query += f" AND Type: {type}"
        issues = self.__client.issues.find(query)
        return [issue.key for issue in issues]

    # Issue attachments
    @log_it
    def get_issue_attachments(self, issue_id: str) -> List[Dict[str, Any]]:
        attachments = list(map(lambda attachment: {
            "name": attachment.name,
            "content": attachment.content,
            "size": attachment.size,
            "mime_type": attachment.mimetype,
            "author": attachment.createdBy.id,
            "created_at": attachment.createdAt,
            "id": attachment.id
        }, self.__client.issues[issue_id].attachments.get_all()))
        return attachments

    @log_it
    def get_issue_attachment(self, issue_id: str, attachment_id: str) -> Dict[str, Any]:
        attachment = self.__client.issues[issue_id].attachments[attachment_id]
        return {
            "name": attachment.name,
            "content": attachment.content,
            "size": attachment.size,
            "mime_type": attachment.mimetype,
            "author": attachment.createdBy.id,
            "created_at": attachment.createdAt,
            "id": attachment.id
        }

    @log_it
    def download_issue_attachments(self, issue_id: str, path: str = './') -> None:
        for attachment in self.__client.issues[issue_id].attachments.get_all():
            attachment.download_to(path)
            logger.info(f'Attachment {attachment.name} downloaded successfully')

    @log_it
    def download_issue_attachment(self, issue_id: str, attachment_id: str, path: str = './') -> None:
        attachment = self.__client.issues[issue_id].attachments[attachment_id]
        attachment.download_to(path)
        logger.info(f'Attachment {attachment.name} downloaded successfully')

    @log_it
    def add_issue_attachment(self, issue_id: str, path_to_file: str) -> None:
        self.__client.issues[issue_id].attachments.create(path_to_file)

    @log_it
    def delete_issue_attachment(self, issue_id: str, attachment_id: Optional[str] = None,
                                attachment_name: Optional[str] = None) -> None:
        if attachment_name is not None and attachment_id is None:
            attachments = self.get_issue_attachments(issue_id=issue_id)
            filtered_attachments = list(filter(lambda attachment: attachment['name'] == attachment_name, attachments))
            if len(filtered_attachments) > 0:
                attachment_id = filtered_attachments[0]["id"]
            else:
                raise ValueError(f'There is no attachments with name {attachment_name}')

        self.__client.issues[issue_id].attachments[attachment_id].delete()

    @log_it
    def upload_attachment(self, path_to_file: str) -> None:
        self.__client.attachments.create(path_to_file)

    # Bulk operations
    @log_it
    def bulk_change_issues_status(self, issue_ids: List[str], transition: str,
                                  resolution: str, comment: str = None, notify: bool = False) -> str:
        if comment is None:
            bulk_change = self.__client.bulkchange.transition(issues=issue_ids,
                                                              transition=transition,
                                                              resolution=resolution)
        else:
            bulk_change = self.__client.bulkchange.transition(issues=issue_ids,
                                                              transition=transition,
                                                              resolution=resolution,
                                                              comment=comment)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_field(self, issue_ids: List[str], field: str, action: str, value) -> str:
        params = {field: {action: value}}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_tags(self, issue_ids: List[str], action: str, tags: List[str]) -> str:
        params = {"tags": {action: tags}}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_assignee(self, issue_ids: List[str], action: str, assignee: str) -> Resource:
        params = {"assignee": {action: assignee}}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_type(self, issue_ids: List[str], type: str) -> Resource:
        params = {"type": type}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_priority(self, issue_ids: List[str], priority: str) -> Resource:
        params = {"priority": priority}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_fix_version(self, issue_ids: List[str], action: str,
                                       version_ids: List[int] = None, version_names: List[str] = None) -> Resource:
        if version_ids is None and version_names is not None:
            queue_versions = self.get_queue_versions(queue=issue_ids[0].split('-')[0])
            version_ids = []
            for name in version_names:
                version_ids.append(list(filter(lambda version: version['name'] == name, queue_versions))[0].id)
        params = {"fixVersions": {action: version_ids}}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_affected_version(self, issue_ids: List[str], action: str, version_ids: List[int] = None,
                                            version_names: List[str] = None) -> Resource:
        if version_ids is None and version_names is not None:
            queue_versions = self.get_queue_versions(queue=issue_ids[0].split('-')[0])
            version_ids = []
            for name in version_names:
                version_ids.append(list(filter(lambda version: version['name'] == name, queue_versions))[0].id)
        params = {"affectedVersions": {action: version_ids}}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_update_issues_component(self, issue_ids: List[str], action: str, component_ids: List[int] = None,
                                     component_names: List[str] = None) -> Resource:
        if component_ids is None and component_names is not None:
            queue_components = self.get_queue_components(queue=issue_ids[0].split('-')[0])
            component_ids = []
            for name in component_names:
                component_ids.append(list(filter(lambda component: component['name'] == name, queue_components))[0].id)
        params = {"components": {action: component_ids}}
        bulk_change = self.__client.bulkchange.update(issues=issue_ids, **params)
        bulk_change.wait()
        return bulk_change.status

    @log_it
    def bulk_move_issues(self, issue_ids: List[str], queue: str) -> str:
        bulk_change = self.__client.bulkchange.move(issue_ids, queue)
        bulk_change.wait()
        return bulk_change.status

    # Changelog
    @log_it
    def get_changelog(self, issue_id: str, sort: str = 'asc', fields: tuple = (), types: tuple = ()) -> List[Resource]:
        return list(self.__client.issues[issue_id].changelog.get_all(sort=sort, field=fields, type=types))

    @log_it
    def get_status_changes(self, issue_id: str, frm: str = None, to: str = None) -> List:
        changelog = self.__client.issues[issue_id].changelog.get_all(field='status')
        result = []
        for change in changelog:
            item = {'date': change.updatedAt, 'user': change.updatedBy.id}
            for field in change.fields:
                if field['field'].id == 'status':
                    item['from'] = field['from'].key if field['from'] is not None else None
                    item['to'] = field['to'].key if field['to'].key else None
            result.append(item)
        if frm:
            result = list(filter(lambda change: change['from'] == frm, result))
        if to:
            result = list(filter(lambda change: change['to'] == to, result))

        return result

    # Priorities
    @log_it
    def get_priorities(self, fields: Optional[List[str]] = None) -> Resource:
        priorities = self.__client.priorities.get_all()
        if fields is not None:
            priorities = list(map(lambda priority: {field: priority[field] for field in fields}, priorities))
        return priorities

    @log_it
    def get_priority(self, priority_id: int, fields: Optional[List[str]] = None) -> Resource:
        priority = self.__client.priorities[priority_id]
        if fields is not None:
            priority = {field: priority[field] for field in fields}
        return priority

    @log_it
    def create_priority(self, key: str, order: int, name_en: str, name_ru: str, description: str) -> Resource:
        priority = self.__client.priorities.create(order=order,
                                                   key=key,
                                                   name={"en": name_en, "ru": name_ru},
                                                   description=description)
        return priority

    @log_it
    def update_priority(self, **kwargs) -> Resource:
        priority = self.__client.priorities.update(kwargs)
        return priority

    # Transitions
    @log_it
    def get_transitions(self, issue_id: str, fields: Optional[List[str]] = None) -> Resource:
        transitions = self.__client.issues[issue_id].transitions.get_all()
        if fields is not None:
            transitions = list(map(lambda transition: {field: transition[field] for field in fields}, transitions))
        return transitions

    # Issue types
    @log_it
    def get_tracker_issue_types(self, fields: Optional[List[str]] = None) -> Resource:
        issue_types = self.__client.issue_types.get_all()
        if fields is not None:
            issue_types = list(map(lambda issue_type: {field: issue_type[field] for field in fields}, issue_types))
        return issue_types

    @log_it
    def get_tracker_issue_type(self, type_id: int, fields: Optional[List[str]] = None) -> Resource:
        issue_type = self.__client.issue_types[type_id]
        if fields is not None:
            issue_type = {field: issue_type[field] for field in fields}
        return issue_type

    @log_it
    def create_tracker_issue_type(self, key: str, name_en: str, name_ru: str, description: str) -> Resource:
        issue_type = self.__client.issue_types.create(key=key,
                                                      name={"en": name_en, "ru": name_ru},
                                                      description=description)
        return issue_type

    @log_it
    def update_tracker_issue_type(self, **kwargs) -> Resource:
        issue_type = self.__client.issue_types.update(kwargs)
        return issue_type

    # Resolutions
    @log_it
    def get_resolutions(self, fields: Optional[List[str]] = None) -> Resource:
        resolutions = self.__client.resolutions.get_all()
        if fields is not None:
            resolutions = list(map(lambda resolution: {field: resolution[field] for field in fields}, resolutions))
        return resolutions

    @log_it
    def get_resolution(self, resolution_id: int, fields: Optional[List[str]] = None) -> Resource:
        resolution = self.__client.resolutions[resolution_id]
        if fields is not None:
            resolution = {field: resolution[field] for field in fields}
        return resolution

    @log_it
    def create_resolution(self, key: str, name_en: str, name_ru: str, description: str) -> Resource:
        resolution = self.__client.resolutions.create(key=key,
                                                      name={"en": name_en, "ru": name_ru},
                                                      description=description)
        return resolution

    @log_it
    def update_resolution(self, **kwargs) -> Resource:
        resolution = self.__client.resolutions.update(kwargs)
        return resolution

    # Statuses
    @log_it
    def get_statuses(self, fields: Optional[List[str]] = None) -> Resource:
        statuses = self.__client.statuses.get_all()
        if fields is not None:
            statuses = list(map(lambda status: {field: status[field] for field in fields}, statuses))
        return statuses

    @log_it
    def get_status(self, status_id: int, fields: Optional[List[str]] = None) -> Resource:
        status = self.__client.statuses[status_id]
        if fields is not None:
            status = {field: status[field] for field in fields}
        return status

    @log_it
    def create_status(self, key: str, name_en: str, name_ru: str, description: str) -> Resource:
        status = self.__client.statuses.create(key=key,
                                               name={"en": name_en, "ru": name_ru},
                                               description=description)
        return status

    @log_it
    def update_status(self, **kwargs) -> Resource:
        status = self.__client.statuses.update(kwargs)
        return status

    # Queue
    @log_it
    def get_queues(self) -> List:
        return self.__client.queues.get_all()

    @log_it
    def get_queue(self, queue: str) -> List:
        return self.__client.queues[queue]

    @log_it
    def create_queue(self, **kwargs) -> None:
        self.__client.queues.create(**kwargs)
        logger.info(f'Queue was created successfully')

    @log_it
    def update_queue(self, queue: str, **kwargs) -> None:
        self.__client.queues[queue].update(**kwargs)
        logger.info(f'Queue {queue} was updated successfully')

    @log_it
    def delete_queue(self, queue: str) -> None:
        self.__client.queues[queue].delete()
        logger.info(f'Queue {queue} was deleted successfully')

    @log_it
    def get_queue_versions(self, queue: str, fields: Optional[List[str]] = None) -> List[Dict]:
        versions = self.__client.queues[queue].versions
        if fields is not None:
            versions = list(map(lambda version: {field: version[field] for field in fields}, versions))
        return versions

    @log_it
    def get_queue_components(self, queue: str, fields: Optional[List[str]] = None) -> List[Dict]:
        components = self.__client.queues[queue].components
        if fields is not None:
            components = list(map(lambda component: {field: component[field] for field in fields}, components))
        return components

    @log_it
    def get_queue_issue_types(self, queue: str, fields: Optional[List[str]] = None) -> List[Dict]:
        issue_types = self.__client.queues[queue].issuetypes
        if fields is not None:
            issue_types = list(map(lambda issue_type: {field: issue_type[field] for field in fields}, issue_types))
        return issue_types

    @log_it
    def get_queue_projects(self, queue: str, fields: Optional[List[str]] = None) -> List[Dict]:
        projects = self.__client.queues[queue].projects
        if fields is not None:
            projects = list(map(lambda project: {field: project[field] for field in fields}, projects))
        return projects

    # Projects
    @log_it
    def get_projects(self, fields: Optional[List[str]] = None) -> List[Dict]:
        projects = self.__client.projects.get_all()
        if fields is not None:
            projects = list(map(lambda project: {field: project[field] for field in fields}, projects))
        return projects

    # Components
    @log_it
    def get_components(self) -> List:
        return self.__client.components.get_all()

    @log_it
    def get_component(self, component_id: int) -> Resource:
        return self.__client.components[component_id]

    @log_it
    def create_component(self, queue: str, name: str, description: Optional[str] = None,
                         lead: Optional[str] = None) -> None:
        self.__client.components.create(queue=queue,
                                        name=name,
                                        lead=lead,
                                        description=description)
        logger.info(f'Component "{name}" was created successfully')

    @log_it
    def delete_component(self, id: Optional[int] = None, queue: Optional[str] = None,
                         name: Optional[str] = None) -> None:
        if id is None and name is not None and queue is not None:
            components = self.get_queue_components(queue=queue)
            id = list(filter(lambda component: component['name'] == name, components))[0].id

        self.__client.components[id].delete()
        logger.info(f'Component {id} was deleted successfully')

    @log_it
    def update_component(self, id: Optional[int] = None, queue: Optional[str] = None,
                         name: Optional[str] = None, new_name: Optional[str] = None,
                         description: Optional[str] = None, lead: Optional[str] = None) -> None:
        data = {'name': new_name, 'description': description, 'lead': lead}
        data = {k: v for k, v in data.items() if v is not None}
        if id is None and name is not None and queue is not None:
            components = self.get_queue_components(queue=queue)
            component_with_name = list(filter(lambda component: component['name'] == name, components))
            if len(component_with_name) > 0:
                id = component_with_name[0].id
            else:
                raise ValueError(f'There is no component with name {name} in {queue} queue')
        self.__client.components[id].update(**data)
        logger.info(f'Component {id} was updated successfully')

    # Versions
    @log_it
    def get_versions(self) -> List:
        return self.__client.versions.get_all()

    @log_it
    def get_version(self, version_id: int) -> Resource:
        return self.__client.versions[version_id]

    @log_it
    def create_version(self, queue: str, name: str, description: Optional[str] = None,
                       start_date: Optional[str] = None, due_date: Optional[str] = None) -> None:
        self.__client.versions.create(queue=queue,
                                      name=name,
                                      startDate=start_date,
                                      dueDate=due_date,
                                      description=description)
        logger.info(f'Version "{name}" was created successfullly')

    @log_it
    def delete_version(self, id: Optional[int] = None, queue: Optional[str] = None, name: Optional[str] = None) -> None:
        if id is None and name is not None and queue is not None:
            versions = self.get_queue_versions(queue=queue)
            id = list(filter(lambda version: version['name'] == name, versions))[0].id
        self.__client.versions[id].delete()
        logger.info(f'Version {id} was deleted successfully')

    @log_it
    def update_version(self, id: Optional[int] = None, queue: Optional[str] = None,
                       name: Optional[str] = None, new_name: Optional[str] = None,
                       description: Optional[str] = None, start_date: Optional[str] = None,
                       due_date: Optional[str] = None) -> None:
        data = {'name': new_name, 'description': description, 'startDate': start_date, 'dueDate': due_date}
        data = {k: v for k, v in data.items() if v is not None}
        if id is None and name is not None and queue is not None:
            versions = self.get_queue_versions(queue=queue)
            version_with_name = list(filter(lambda version: version['name'] == name, versions))
            if len(version_with_name) > 0:
                id = version_with_name[0].id
            else:
                raise ValueError(f'There is no version with name {name} in {queue} queue')
        self.__client.versions[id].update(**data)
        logger.info(f'Version {id} was updated successfully')

    # Screens
    @log_it
    def get_screens(self) -> List:
        return self.__client.screens.get_all()

    # Sprints
    @log_it
    def get_sprint(self, sprint_id: int) -> Dict:
        return self.__client.sprints[sprint_id]

    @log_it
    def get_sprints_by_board(self, board_id: int) -> List:
        return self.__client.boards[board_id].sprints

    @log_it
    def create_sprint(self, name: str, board_id: int, start_date: str, end_date: str) -> Dict:
        return self.__client.sprints.create(name=name,
                                            board={'id': board_id},
                                            startDate=start_date,
                                            endDate=end_date)
