from dataclasses import dataclass
from furl import furl

import logging
import os
import requests
from requests.adapters import HTTPAdapter
from requests.exceptions import ConnectionError, HTTPError
import urllib3
import time
from startrek_client import Startrek


urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
logging.basicConfig(format="[%(asctime)s | %(levelname)s]: %(message)s",
                    datefmt="%m.%d.%Y %H:%M:%S",
                    level=logging.INFO)

@dataclass
class Data:
    name = os.getenv('ROBOT_NAME', '')
    token = os.getenv('TRACKER_TOKEN', '')
    url = 'https://st-api.yandex-team.ru'
    custom_filter = 'Queue: MAILPRODUCT AND Status: !Closed'
    tag = 'need_checklist'

@dataclass
class LightMode:
    experiment = 'Light'
    checklist = ['Тикет MAILPRODUCT: проверить описание',
               'Тикет MAILPRODUCT: проверить наличие гипотезы',
               'Тикет MAILPRODUCT: Описать эксперимент и какие выборки нужны.',
               'Эксперимент: Указать метрики для проверки эксперимента.',
               'Написать письмо о начале эксперимента(mail - experiments @)',
               'Спустя день запустить расчет метрик',
               'Проверить прокрас метрик',
               'Запустить расчет метрик',
               'В тикете приложить расчет метрик и призвать аналитик (дежурного)',
               'Аналитик в течении трех рабочих дней отписывается о результатах (либо просит еще три, и через три дня повторно отписывается). Результат должен быть подробно расписан, что означает изменение цифр.',
               'Принять решение об эксперименте.',
               'Призвать EE по деньгам и получить ОК от него (vkokarev@)',
               'Удаление эксперимента']

@dataclass
class HardMode:
    experiment = 'Hard'
    checklist = ['Тикет MAILPRODUCT: проверить описание',
               'Тикет MAILPRODUCT: проверить наличие гипотезы',
               'Тикет MAILPRODUCT: проверить наличие обоснования гипотезы (SBS, UX). Если их нет, написать причину их отсутствия',
               'Тикет MAILPRODUCT: Описать эксперимент и какие выборки нужны.',
               'Эксперимент: Указать метрики для проверки эксперимента.',
               'Эксперимент: Просмотр дизайна нового функционала совместно с аналитиком',
               'Эксперимент: если нет метрик - завести тикет в MAILAN',
               'Эксперимент: Призвать аналитика для проверки данных (который ведет эксперимент или EE plum@)',
               'Написать письмо о начале эксперимента(mail - experiments @)',
               'Спустя день запустить расчет метрик',
               'Проверить прокрас метрик',
               'Запустить расчет метрик',
               'В тикете приложить расчет метрик и призвать аналитик (дежурного)',
               'Аналитик в течении трех рабочих дней отписывается о результатах (либо просит еще три, и через три дня повторно отписывается). Результат должен быть подробно расписан, что означает изменение цифр.',
               'Принять решение об эксперименте.',
               'Призвать EE по деньгам и получить ОК от него (vkokarev@)']


class Session:
    def __init__(self, key):
        http_adapter = HTTPAdapter(max_retries=5)
        self.session = requests.Session()
        self.session.mount('https://', http_adapter)
        self.session.mount('http://', http_adapter)

        self.session.headers['Content-Type'] = 'application/json'
        self.session.headers['Authorization'] = f"OAuth {Data.token}"
        self.session.headers['User-Agent'] = 'curl/7.53.1'
        self.session.headers['Connection'] = 'close'
        self.session.headers['Accept'] = '*/*'

        self.url = furl(f'{Data.url}/v2/issues/{key}/checklistItems')

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.session.close()


def create_checklist(key, data):
    with Session(key) as st_process:
        try:
            logging.info(f'Ticket {key}: Create checklist for ticket...')
            response = st_process.session.post(st_process.url, json=data, verify=False)
            response.raise_for_status()
            logging.info(f'Ticket {key}: Checklist was created for ticket.')
        except HTTPError as http_err:
            logging.info(f'Ticket {key}: Checklist wasn\'t created.'
                         f'Response = [status code = {response.status_code}, text = {response.text}].'
                         f'HTTP error occurred: {http_err}')
        except Exception as err:
            logging.info(f'Ticket {key}: Checklist wasn\'t created.'
                         f'Response = [status code = {response.status_code}, text = {response.text}].'
                         f'HTTP error occurred: {err}')
        except ConnectionError as c_err:
            logging.info(f'Ticket {key}: Checklist wasn\'t created.'
                         f'Response = [status code = {response.status_code}, text = {response.text}].'
                         f'HTTP error occurred: {c_err}')


def main():
    client = Startrek(base_url=Data.url,
                      useragent=Data.name,
                      token=Data.token)



    # Add checklist light
    filter = f'{Data.custom_filter} AND Tags:{Data.tag} AND Experiment:{LightMode.experiment}'
    issues = client.issues.find(filter)
    data = [{"checked": False, "text": item} for item in list(LightMode.checklist)]

    while len(issues):
        for ticket in issues:
            create_checklist(ticket.key, data)
            item = client.issues[ticket.key]
            item.update(tags={'remove': [Data.tag]}, ignore_version_change=True)
            logging.info(f'Tag {Data.tag} delete for ticket {item.key}')
        time.sleep(7)
        issues = client.issues.find(filter)

    # Add checklist hard
    filter = f'{Data.custom_filter} AND Tags:{Data.tag} AND Experiment:{HardMode.experiment}'
    issues = client.issues.find(filter)
    data = [{"checked": False, "text": item} for item in list(HardMode.checklist)]

    while len(issues):
        for ticket in issues:
            create_checklist(ticket.key, data)
            item = client.issues[ticket.key]
            item.update(tags={'remove': [Data.tag]}, ignore_version_change=True)
            logging.info(f'Tag {Data.tag} delete for ticket {item.key}')
        time.sleep(7)
        issues = client.issues.find(filter)


if __name__ == '__main__':
    main()
