import json
import logging
import os
from dataclasses import dataclass
from enum import Enum, auto
from typing import Dict, Optional, List, Any

import requests
from furl import furl
from requests import Response
from requests.adapters import HTTPAdapter
from urllib3 import Retry

logger = logging.getLogger('wiki_logger')


@dataclass
class Constants:
    base_url = 'https://wiki-api.yandex-team.ru'
    url_path_prefix = ['_api', 'frontend']


@dataclass
class TableChangeType:
    column_moved = 'column_moved'
    edited_row = 'edited_row'
    row_moved = 'row_moved'
    added_column = 'added_column'
    added_row = 'added_row'
    edited_column = 'edited_column'
    removed_column = 'removed_column'
    removed_row = 'removed_row'
    title_changed = 'title_changed'
    soring_changed = 'soring_changed'


class RowPosition(Enum):
    first = '-1'
    last = auto()


def log_it(method):
    def wrapper(self, *args, **kwargs):
        logger.debug(f'Run function {method.__name__} with arguments {kwargs}')
        res = method(self, *args, **kwargs)
        logger.debug(f'Result: {res}')
        return res

    return wrapper


class WikiClient:
    def __init__(self, auth, host=Constants.base_url):
        self.__auth = auth
        self.__host = host
        self.__headers = {
            'Authorization': f'OAuth {self.__auth}',
            'Content-Type': 'application/json'
        }

    @property
    def host(self) -> str:
        return self.__host

    @host.setter
    def host(self, new_host: str) -> None:
        self.__host = new_host

    @staticmethod
    def __connect():
        r = requests.Session()
        retries = Retry(total=5,
                        backoff_factor=0.3,
                        status_forcelist=[500, 502, 503, 504],
                        method_whitelist=frozenset(['GET', 'POST']))
        adapter = HTTPAdapter(max_retries=retries)
        r.mount('https://', adapter)
        r.mount('http://', adapter)
        r.verify = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'cacert.pem')
        return r

    @staticmethod
    def __serialize_to_json(data: Dict) -> Optional[bytes]:
        if data is None:
            return data
        return json.dumps(data, ensure_ascii=False, separators=(',', ': ')).encode('utf-8')

    def __update_headers_if_needed(self, headers: Optional[Dict[str, str]] = None) -> Dict[str, str]:
        new_headers = self.__headers.copy()
        if headers:
            for header, value in headers.items():
                new_headers[header] = value
        return new_headers

    @log_it
    def __make_get_request(self, url: str) -> Response:
        response = self.__connect().get(url, headers=self.__headers)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_post_request(self, url: str, data=None, files=None) -> Response:
        if files is not None:
            del self.__headers["Content-Type"]
        response = self.__connect().post(url, headers=self.__headers, data=self.__serialize_to_json(data), files=files)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_delete_request(self, url: str, data=None, headers: Optional[Dict[str, str]] = None) -> Response:
        headers = self.__update_headers_if_needed(headers)
        response = self.__connect().delete(url, data=self.__serialize_to_json(data), headers=headers)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def __make_put_request(self, url: str, data, headers: Optional[Dict[str, str]] = None) -> Response:
        headers = self.__update_headers_if_needed(headers)
        response = self.__connect().put(url, headers=headers, data=data)
        logger.debug(response.text)
        response.raise_for_status()
        return response

    @log_it
    def get_page_attributes(self, path: List[str]) -> json:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + path)
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def get_table_structure(self, path: List[str]) -> json:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + path + ['.grid'])
        response = self.__make_get_request(url)
        return response.json()

    @log_it
    def edit_table_structure(self, path: List[str], change_type: str, **kwargs) -> json:
        url = furl(self.__host) \
            .add(path=Constants.url_path_prefix + path + ['.grid', 'change'])
        version_number = self.get_table_structure(path)['data']['version']
        data = {
            "version": version_number,
            "changes": [{
                change_type: kwargs
            }]
        }
        response = self.__make_post_request(url, data)
        return response.json()

    @log_it
    def add_table_row(self, path: List[str], row_position: Any, data=None, **kwargs) -> json:
        if data is None:
            data = {}

        table = self.get_table_structure(path)
        if row_position == RowPosition.first:
            after_id = '-1'
        elif row_position == RowPosition.last:
            after_id = table['data']['rows'][-1][0]['row_id']
        else:
            after_id = table['data']['rows'][row_position-2][0]['row_id']

        return self.edit_table_structure(path=path, change_type=TableChangeType.added_row, after_id=after_id, data=data, **kwargs)

    @log_it
    def remove_table_row(self, path: List[str], row_position: Any) -> json:
        table = self.get_table_structure(path)

        if row_position == RowPosition.first:
            row_id = table['data']['rows'][0][0]['row_id']
        elif row_position == RowPosition.last:
            row_id = table['data']['rows'][-1][0]['row_id']
        else:
            row_id = table['data']['rows'][row_position][0]['row_id']

        return self.edit_table_structure(path=path, change_type=TableChangeType.removed_row, id=row_id)
