#!/usr/bin/python3
# -*- coding: UTF-8 -*-

import argparse
import json
import os
import re
import subprocess
import sys
import time


DEFAULT_LAG_LIMIT = 524288 #0.5 MB
DEFAULT_DELAY_CONFIG = "/etc/push-client/monitoring.json"
DEFAULT_CFG_PATH = "/etc/push-client/"


def report(status, info):
    print("%d;%s" % (status, info))
    exit()


def get_push_client_path():
    cmd = "which push-client"
    p = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE)
    push_client_path = p.communicate()[0].decode("utf-8").split("\n")[0]
    return push_client_path


def get_push_client_pids(app):
    cmd = "pidof " + app
    p = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE)
    pids = p.communicate()[0].decode("utf-8").split("\n")[0]
    return pids.split()


def get_state_by_config(app, config):
    cmd = app + " --status --json -c " + config
    with open(os.devnull, "w") as FNULL:
        p = subprocess.Popen(
            cmd.split(), stdout=subprocess.PIPE, stderr=FNULL)
    info, out = p.communicate()
    return json.loads(info.decode("utf-8"))


def push_client_launched_configs(pids):
    pids = " ".join(pids)
    cmd = "ps " + pids
    p = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE)
    ps_out = p.communicate()[0].decode("utf-8").split()

    configs = []
    for item in ps_out:
        if re.search(".yaml", item) or re.search(".yml", item):
            configs.append(item)
    configs = list(set(configs))
    return configs


def push_client_configs():
    return [os.path.join(DEFAULT_CFG_PATH, f)
            for f in os.listdir(DEFAULT_CFG_PATH)
            if os.path.isfile(os.path.join(DEFAULT_CFG_PATH, f))]


def load_delay_config(config):
    """ Config example:
        {
            "/etc/push-client/smtp-sologstore.yaml": {
                "/var/log/nwsmtp-tskv/nwtskv.log": 1000
            }
        }
    """
    try:
        with open(config) as f:
            return json.loads(f.read())
    except IOError:
        return {}


def main(args):
    push_client_path = get_push_client_path()
    if push_client_path == "":
        report(2, "can't find push-client")

    pids = get_push_client_pids(push_client_path)
    if len(pids) == 0:
        report(2, "push-client stopped")

    configs = set(push_client_launched_configs(pids) + push_client_configs())

    current_time = int(time.time())

    delay_config = load_delay_config(args.config)

    problems = []

    for config in configs:
        config_name = re.sub(".*/", "", config)

        config_logs = get_state_by_config(push_client_path, config)

        if len(config_logs) == 0:
            msg = "%s: empty state" % (config_name)
            problems.append(msg)

        for log in config_logs:
            if log["type"] == "missing state":
                msg = "%s: %s: missing state" % (config_name, log["name"])
                problems.append(msg)
            elif log["name"] == log["actual name"]:
                log_name = log["name"]
                last_commit_time = int(log["last commit time"])


                if log["lag"] > args.lag_limit:
                    msg = "%s: %s:%d bytes" % (
                        config_name, log_name, commit_delay)
                    problems.append(msg)

    if len(problems) > 0:
        report(2, ";  ".join(problems))
    report(0, "OK")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description=(
        "Push-client status and lag monitoring"))
    parser.add_argument("-c", "--config", type=str,
                        help="Path to custom delay config", required=False, default=DEFAULT_DELAY_CONFIG)
    parser.add_argument("-l", "--lag-limit", type=int, default=DEFAULT_LAG_LIMIT,
                        help="Critical lag of log (in bytes)", required=False)
    args = parser.parse_args()
    main(args)
