import re
from collections import defaultdict
from urlparse import urlparse
from mail.monitoring.common.timetail import timetail
from mail.monitoring.delivery.lib import yasm


def get_code_ranges(code_ranges):
    """
    >>> r = get_code_ranges('0:2,5:7,9:13')
    >>> list(r)
    [0, 1, 2, 5, 6, 7, 9, 10, 11, 12, 13]
    """
    code_ranges = code_ranges.split(",")
    ranges = []
    for code_range in code_ranges:
        code_from, code_to = tuple(code_range.split(":"))
        ranges.extend(xrange(int(code_from), int(code_to) + 1))
    return ranges


def parse_url_query(query):
    return dict(field.split("=", 1) for field in query.split("&") if "=" in field)


def strip_url(url_path):
    url_path = url_path.replace("/", "_")
    url_path = url_path.replace(".", "_")
    url_path = url_path.split(":")[0]    # strorage requests hack
    url_path = url_path.split("320")[0]  # strorage requests hack
    url_path = re.sub("^_", "", url_path)
    return url_path


def make_default_http_client_recognizer():
    session_service_re = re.compile(
        r"(?P<connection_id>[^-\s]+)-(?P<envelope_id>[^-\s]+)-(?P<service>\w+)")

    def recognize(pairs):
        match = session_service_re.match(pairs.get("y_context", ""))
        return match.group("service") if match else "_UNKNOWN"

    return recognize


def make_get_http_client_info(http_client_recognizer):
    if http_client_recognizer is None:
        http_client_recognizer = make_default_http_client_recognizer()

    def get_http_client_info(log_file, time=60):
        http_stat = defaultdict(lambda: defaultdict(lambda: {"http_codes": defaultdict(int), "timings": []}))
        for pairs in timetail(log_file=log_file, fmt="tskv", time=time):
            service = http_client_recognizer(pairs)
            if service is not None:
                url = urlparse("http://" + pairs["uri"])
                url_path = strip_url(url.path)
                http_code = int(pairs["status"])
                timing = float(pairs["total_time"])
                http_stat[service]["_all"]["http_codes"][http_code] += 1
                http_stat[service]["_all"]["timings"].append(timing)
                http_stat[service][url_path]["http_codes"][http_code] += 1
                http_stat[service][url_path]["timings"].append(timing)
        return http_stat

    return get_http_client_info


def get_web_server_info(log_file, time=60):
    http_stat = defaultdict(lambda: defaultdict(lambda: {"http_codes": defaultdict(int), "timings": []}))
    for pairs in timetail(log_file=log_file, fmt="tskv", time=time):
        url = urlparse(pairs["request"])
        url_path = strip_url(url.path)
        http_code = int(pairs["status_code"])
        timing = float(pairs["profiler_exec"])
        service = parse_url_query(url.query).get("service", None)
        http_stat["_all"]["_all"]["http_codes"][http_code] += 1
        http_stat["_all"]["_all"]["timings"].append(timing)
        http_stat["_all"][url_path]["http_codes"][http_code] += 1
        http_stat["_all"][url_path]["timings"].append(timing)
        if service:
            http_stat[service]["_all"]["http_codes"][http_code] += 1
            http_stat[service]["_all"]["timings"].append(timing)
            http_stat[service][url_path]["http_codes"][http_code] += 1
            http_stat[service][url_path]["timings"].append(timing)
    return http_stat


def yasm_send_http_stat_custom(log_path, app, http_type=None, http_client_recognizer=None):
    if re.search("access", log_path) or http_type == "web_server":
        http_type = "web_server"
        get_http_stat = get_web_server_info
    if re.search("http_client", log_path) or http_type == "http_client":
        http_type = "http_client"
        get_http_stat = make_get_http_client_info(http_client_recognizer=http_client_recognizer)
    http_stat = get_http_stat(log_path)
    for service, service_info in http_stat.iteritems():
        request_codes = {}
        request_timings = {}
        for url, request_data in service_info.iteritems():
            for code, count in request_data["http_codes"].iteritems():
                request_codes_key = "%s_%s" % (url, code)
                request_codes[request_codes_key] = count
            request_timings[url] = request_data["timings"]
        yasm.send(request_codes, prefix=http_type + "_code", suffix="tmmx", itype=app, ctype=service, debug=True)
        yasm.send(request_timings, prefix=http_type + "_timings", suffix="ahhh", itype=app, ctype=service)


def yasm_send_http_stat(log_path, app, http_type=None):
    return yasm_send_http_stat_custom(log_path, app, http_type)
