import re
import requests
from collections import defaultdict
from mail.monitoring.common.timetail import timetail

LOG_FILE = "/app/log/mdbsave.tskv"
HTTP_CLIENT_LOG_FILE = "/app/log/http_client.tskv"
TIMETAIL_FMT = "tskv"
SHARPEI_STAT_PATH = "/v3/stat"
SHARPEI_HOSTS = {"bigmail": "sharpei-production.mail.yandex.net", "corp": "sharpei-intranet-production.mail.yandex.net"}
SHARPEI_TIMEOUT = 2


def get_services():
    services = {}
    for host in SHARPEI_HOSTS.values():
        services[host] = "sharpei"
    return services


def get_sharpei_host_per_shard(sharpei_host):
    host_shard = {}
    sharpei_data = requests.get("http://" + sharpei_host + SHARPEI_STAT_PATH, timeout=SHARPEI_TIMEOUT).json()

    for shard_data in sharpei_data.values():
        shard_name = shard_data["name"]
        for host in shard_data["databases"]:
            host = host["address"]["host"]
            host_shard[host] = shard_name
    return host_shard


def strip_underscore(metric):
    metric = re.sub(',.*', '', metric)
    return re.sub(' ', '_', metric)


def get_errors_count_per_shard(time=60, get_logs=timetail):
    host_shard = {}
    for sharpei_host in SHARPEI_HOSTS.values():
        host_shard.update(get_sharpei_host_per_shard(sharpei_host))
    failed_shards = defaultdict(int)
    failed_shard_re = re.compile(
        r"query failed with.*host=(?P<host>\S+)")
    for line in get_logs(LOG_FILE, time=time, fmt=TIMETAIL_FMT):
        if line["level"] != "error":
            continue
        if "message" not in line:
            continue
        match = failed_shard_re.search(line["message"])
        if match:
            shard = host_shard.get(match.group("host"))
            failed_shards[shard] += 1
    return failed_shards


def get_db_errors(time=60, get_logs=timetail):
    db_errors = defaultdict(int)
    db_error_re = re.compile("(?P<db_error>.*?): ")
    for line in get_logs(LOG_FILE, time=time, fmt=TIMETAIL_FMT):
        if line["level"] != "error":
            continue
        if "error_type" not in line or line["error_type"] != "macs_error":
            continue
        match = db_error_re.search(line["error_code.message"])
        if match:
            db_error = strip_underscore(match.group("db_error"))
            db_errors[db_error] += 1
    return db_errors


def get_success_save_count(time=60, get_logs=timetail):
    save_success_count = 0
    re_success_save = re.compile("save success")
    for line in get_logs(LOG_FILE, time=time, fmt=TIMETAIL_FMT):
        if "message" in line:
            if re_success_save.search(line["message"]):
                save_success_count += 1
    return save_success_count


def _get_http_client_stat(services, log, timefmt, time, get_logs=timetail):
    info = defaultdict(lambda: {
        "http_codes": defaultdict(int),
        "timings": []
    })

    # tskv	tskv_format=mail-http_client-format-log	timestamp=2021-09-12T22:52:48.219877+03:00	session_id=0x145656470
    # y_context=EJB6ap9HvM-A1bRrVes	host=sharpei-production.mail.yandex.net	port=80
    # uri=sharpei-production.mail.yandex.net:80/conninfo?mode=all&uid=76600470	status=200	resolve_time=0.000
    # connect_time=0.000	tls_time=0.000	total_time=0.001	error=success	bytes_out=201	bytes_in=374	attempt=0

    for params in get_logs(log, time=time, fmt=timefmt):
        if not all(key in params for key in ("host", "status", "total_time")):
            continue

        srv = services.get(params["host"])
        if srv:
            info[srv]["http_codes"][int(params["status"])] += 1
            info[srv]["timings"].append(float(params["total_time"]))

    return info


def get_http_client_stat(log=HTTP_CLIENT_LOG_FILE, timefmt=TIMETAIL_FMT, time=60, get_logs=timetail):
    return _get_http_client_stat(get_services(), log, timefmt, time, get_logs)
