import re
import sys
from collections import defaultdict
from mail.monitoring.common.timetail import timetail

HTTP_CLIENT_IN_LOG_PATH = "/var/log/nwsmtp/http_client.tskv"
HTTP_CLIENT_OUT_LOG_PATH = "/var/log/nwsmtp/http_client-out.tskv"
NWSMTP_ACCESS_LOG_PATH = "/var/log/nwsmtp/access.tskv"
NWSMTP_IN_LOG_PATH = "/var/log/nwsmtp/nwsmtp.tskv"
NWSMTP_OUT_LOG_PATH = "/var/log/nwsmtp/nwsmtp-out.tskv"
YPLATFORM_LOG_PATH = "/var/log/nwsmtp/yplatform.log"

NWSMTP_APP = "nwsmtp"
NWSMTP_OUT_APP = "nwsmtp-out"
HTTP_CLIENT_LOGS = {NWSMTP_APP: HTTP_CLIENT_IN_LOG_PATH, NWSMTP_OUT_APP: HTTP_CLIENT_OUT_LOG_PATH}
NWSMTP_LOGS = {NWSMTP_APP: NWSMTP_IN_LOG_PATH, NWSMTP_OUT_APP: NWSMTP_OUT_LOG_PATH}

TIMETAIL_FMT = "tskv"


def get_avir_statuses(avir_type, time=60, get_logs=timetail, host=None, app=NWSMTP_APP):
    avir_statuses = defaultdict(int)
    status_re = re.compile(r".*host='(?P<host>.*)'.*status='(?P<status>\w+)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == avir_type:
            match = status_re.search(pairs.get("message", ""))
            if (match is not None) and ((host is None) or (host == match.group("host"))):
                avir_statuses[match.group("status")] += 1
    return avir_statuses


def get_avir_timings(avir_type, time=60, get_logs=timetail, host=None, app=NWSMTP_APP):
    avir_timings = []
    delay_re = re.compile(".*host='(?P<host>.*)'.*delay=(?P<delay>.*), size")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == avir_type:
            match = delay_re.search(pairs.get("message", ""))
            if (match is not None) and ((host is None) or (host == match.group("host"))):
                avir_timings.append(float(match.group("delay")))
    return avir_timings


def get_backup_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    backup_statuses = defaultdict(int)
    status_re = re.compile(r".*status=(?P<status>\w+)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if "-BACK" in pairs.get("where_name", ""):
            match = status_re.search(pairs.get("message", ""))
            if match is not None:
                backup_statuses[match.group('status')] += 1
    return backup_statuses


def get_ml_statuses(time=60, get_logs=timetail, filtered_rcpts=set(), app=NWSMTP_APP):
    ml_statuses = defaultdict(int)
    status_re = re.compile(r"email='(?P<email>[^']+)'.*stat=(?P<status>\w+)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "LIST":
            message = pairs.get("message", "")
            match = status_re.search(message)
            if (match is not None) and (match.group('email') not in filtered_rcpts):
                status = match.group('status')
                if (status == 'error') and (re.search('Not found', message) is not None):
                    status = 'not_found'
                ml_statuses[status] += 1
    return ml_statuses


def get_dkim_sign_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    dkim_statuses = defaultdict(lambda: defaultdict(int))
    # Oct 30 00:00:02 smtp1p nwsmtp: ltsDWmEswT-02omxrM3-DKIM: added a
    # signature for @73237.ru
    ok_part = "added a signature for @"
    # Oct 30 00:00:02 smtp1p nwsmtp: HYd2u9phaR-02o0tnC2-DKIM: dkim: the
    # specified domain not found in dkim key list: palax.info
    miss_part = "dkim: the specified domain not found in dkim key list: "
    dkim_sign_re = re.compile(r"(?P<message>{ok}|{miss})(?P<domain>\S+)".format(ok=ok_part, miss=miss_part))
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "DKIM":
            match = dkim_sign_re.search(pairs.get("message", ""))
            if match is not None:
                if match.group("message") == ok_part:
                    status = "ok"
                else:
                    status = "miss"
                dkim_statuses[match.group("domain")][status] += 1
    return dkim_statuses


def _get_code_message(relay_answer):
    code, msg = 0, ""
    if relay_answer:
        raw_code = relay_answer.split(" ", 1)[0]
        try:
            code = int(raw_code)
        except ValueError:
            pass
        msg = " ".join(relay_answer.split()[1:])
    return code, msg


def _should_ignore(message, ignore_messages):
    # `message` is like "554 5.1.1 Unknown user; t2s3Iys4"
    # `ignore_message` is like "Unknown user"
    for ignore_message in filter(None, map(str.strip, ignore_messages)):
        if ignore_message.lower() in message.lower():
            return True
    return False


def get_relays_statuses(time=60, get_logs=timetail, ignore_messages=(), app=NWSMTP_APP):
    """
    >>> get_relays_statuses(get_logs=lambda log_file, fmt, time: [])["relays"]
    defaultdict(<function <lambda> at ..., {})
    >>> pairs = {'tskv_format': 'mail-nwsmtp-tskv-log',
    ...          'thread': '139778569795328',
    ...          'unixtime': '1594117262',
    ...          'timestamp': '2020-07-07T13:21:02.742966+0300',
    ...          'level': 'notice',
    ...          'connection_id': '4SRzuC3UWb',
    ...          'envelope_id': 'L2b8pA9r',
    ...          'uniq_id': '4SRzuC3UWb-L2b8pA9r',
    ...          'where_name': 'SEND-LOCAL',
    ...          'message': 'to=<fbl-mail@mail.eobrychet-mail.ru>, relay=127.0.0.1:1234, real=127.0.0.1:1234, delay=0.043, '
    ...                     'status=fault (554 5.1.1 Unknown user; 47DozR67at-sVUIG200)',
    ...          'direction': 'in'}
    >>> get_relays_statuses(get_logs=lambda log_file, fmt, time: [pairs])["relays"]
    defaultdict(<function <lambda> at ...>, {'127.0.0.1:1234': defaultdict(<type 'int'>, {554: 1}), 'all_relays': defaultdict(<type 'int'>, {554: 1})})
    >>> pairs = {'tskv_format': 'mail-nwsmtp-tskv-log',
    ...          'thread': '139778569795328',
    ...          'unixtime': '1594117262',
    ...          'timestamp': '2020-07-07T13:21:02.742966+0300',
    ...          'level': 'notice',
    ...          'connection_id': '4SRzuC3UWb',
    ...          'envelope_id': 'L2b8pA9r',
    ...          'uniq_id': '4SRzuC3UWb-L2b8pA9r',
    ...          'where_name': 'SEND-LOCAL',
    ...          'message': 'to=<fbl-mail@mail.eobrychet-mail.ru>, relay=127.0.0.1:1234, real=127.0.0.1:1234, delay=0.043, '
    ...                     'status=fault (554 5.1.1 Unknown user; 47DozR67at-sVUIG200)',
    ...          'direction': 'in'}
    >>> get_relays_statuses(get_logs=lambda log_file, fmt, time: [pairs], ignore_messages=('Unknown user',))["relays"]
    defaultdict(<function <lambda> at ..., {})
    >>> pairs = {'tskv_format': 'mail-nwsmtp-tskv-log',
    ...          'thread': '139778569795328',
    ...          'unixtime': '1594117262',
    ...          'timestamp': '2020-07-07T13:21:02.742966+0300',
    ...          'level': 'notice',
    ...          'connection_id': '4SRzuC3UWb',
    ...          'envelope_id': 'L2b8pA9r',
    ...          'uniq_id': '4SRzuC3UWb-L2b8pA9r',
    ...          'where_name': 'SEND-LOCAL',
    ...          'message': 'to=<no-reply@qoob.su>, relay=mxbacks.mail.yandex.net:27, real=127.0.0.1:1234, delay=0.185, '
    ...                     'status=fault (554 5.7.0 Failed to authorize the sender '
    ...                     '1589879893-9tEEFbaCDx-IDMi44FX)',
    ...          'direction': 'in'}
    >>> get_relays_statuses(get_logs=lambda log_file, fmt, time: [pairs])["relays"]
    defaultdict(<function <lambda> at ...>, {'all_relays': defaultdict(<type 'int'>, {554: 1}), 'mxbacks.mail.yandex.net:27': defaultdict(<type 'int'>, {554: 1})})
    >>> pairs = {'tskv_format': 'mail-nwsmtp-tskv-log',
    ...          'thread': '139778569795328',
    ...          'unixtime': '1594117262',
    ...          'timestamp': '2020-07-07T13:21:02.742966+0300',
    ...          'level': 'notice',
    ...          'connection_id': '4SRzuC3UWb',
    ...          'envelope_id': 'L2b8pA9r',
    ...          'uniq_id': '4SRzuC3UWb-L2b8pA9r',
    ...          'where_name': 'SEND-LOCAL',
    ...          'message': 'to=<no-reply@qoob.su>, relay=mxbacks.mail.yandex.net:27, real=mxbacks.mail.yandex.net:27, delay=0.185, '
    ...                     'status=fault (554 5.7.0 Failed to authorize the sender '
    ...                     '1589879893-9tEEFbaCDx-IDMi44FX)',
    ...          'direction': 'in'}
    >>> get_relays_statuses(get_logs=lambda log_file, fmt, time: [pairs], ignore_messages=('5.7.0 Failed to authorize the sender',))["relays"]
    defaultdict(<function <lambda> at ..., {})
    """
    relays_statuses = defaultdict(lambda: defaultdict(int))
    relays_real_statuses = defaultdict(lambda: defaultdict(lambda: defaultdict(int)))
    status_re = re.compile(r"relay=(?P<relay>.*), real=(?P<real>.*), delay.*status=(?P<status>\w+).*\((?P<relay_answer>.*)\)")
    status_re = re.compile(r"relay=(?P<relay>[^,]*), real=(?P<real>[^,]*), delay=[^,]*, status=(?P<status>[^ ]+) \((?P<relay_answer>[^)]*)\)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        suffixes = ["SEND-SMTP-RELAY", "SEND-LOCAL", "SEND-SMTP", "SEND-NSLS"]
        if any(suffix in pairs.get("where_name", "") for suffix in suffixes):
            match = status_re.search(pairs.get("message", ""))
            if match is not None:
                try:
                    code, message = _get_code_message(match.group("relay_answer"))
                except Exception as e:
                    exc_type, _, tb = sys.exc_info()
                    line = "\t".join(key + "=" + value for key, value in pairs.items())
                    raise exc_type, "Get exc: %s, for line: %s" % (e, line), tb
                if _should_ignore(message, ignore_messages):
                    continue
                relays_statuses[match.group("relay")][code] += 1
                relays_statuses["all_relays"][code] += 1
                relays_real_statuses[match.group("relay")][match.group("real")][code] += 1
    return {"relays": relays_statuses, "relays_real": relays_real_statuses}


def get_so_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    """
    >>> get_so_statuses(get_logs=lambda log_file, fmt, time: [])
    defaultdict(<type 'int'>, {})
    >>> pairs = {'tskv_format': 'mail-nwsmtp-tskv-log',
    ...          'thread': '139778536224512',
    ...          'unixtime': '1594074030',
    ...          'timestamp': '2020-07-07T01:20:30.386594+0300',
    ...          'level': 'notice',
    ...          'connection_id': '0qJhjkCFEW',
    ...          'envelope_id': 'KTbWCLwn',
    ...          'uniq_id': '0qJhjkCFEW-KTbWCLwn',
    ...          'where_name': 'SOCHECK',
    ...          'message': 'from=startrek@yandex-team.ru, '
    ...                     'status=accept, '
    ...                     'so_classes=t_news,personal,s_tracker,schema,trackertask,trust_6',
    ...          'direction': 'in'}
    >>> get_so_statuses(get_logs=lambda log_file, fmt, time: [pairs])
    defaultdict(<type 'int'>, {'accept': 1})
    """
    so_statuses = defaultdict(int)
    status_re = re.compile(r"status=(?P<status>\w+)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "SOCHECK":
            message = pairs.get("message", "")
            match = status_re.search(message)
            if match is not None:
                so_statuses[match.group('status')] += 1
    return so_statuses


def get_rc_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    rc_statuses = defaultdict(int)
    status_re = re.compile(r"status=(?P<status>\w+)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "RC":
            match = status_re.search(pairs.get("message", ""))
            if match is not None:
                rc_statuses[match.group('status')] += 1
    return rc_statuses


def get_fouras_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    fouras_statuses = defaultdict(lambda: defaultdict(int))
    status_re = re.compile(r"status=(?P<status>\w+), reason='(?P<reason>.*)'")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "FOURAS":
            match = status_re.search(pairs.get("message", ""))
            if match is not None:
                fouras_statuses[match.group('status')][match.group('reason')] += 1
    return fouras_statuses


def get_fouras_timings(time=60, get_logs=timetail, app=NWSMTP_APP):
    fouras_timings = []
    delay_re = re.compile(r"delay=(?P<delay>[0-9\.]*)")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "FOURAS":
            match = delay_re.search(pairs.get("message", ""))
            if match is not None:
                fouras_timings.append(float(match.group("delay")))
    return fouras_timings


def get_ratesrv_statuses(time=60, get_logs=timetail):
    ratesrv_statuses = defaultdict(int)
    status_re = re.compile(r"-RATESRV: request size.*status=(?P<status>\w+)")
    for line in get_logs(log_file=YPLATFORM_LOG_PATH, time=time):
        match = status_re.search(line)
        if match is not None:
            ratesrv_statuses[match.group('status')] += 1
    return ratesrv_statuses


def get_ratesrv_timings(time=60, get_logs=timetail):
    ratesrv_timings = []
    delay_re = re.compile(r"-RATESRV: request size.*delay=(?P<delay>[0-9\.]*)")
    for line in get_logs(log_file=YPLATFORM_LOG_PATH, time=time):
        match = delay_re.search(line)
        if match is not None:
            ratesrv_timings.append(float(match.group("delay")))
    return ratesrv_timings


def _get_reject_statuses(event_types, app, time=60, get_logs=timetail):
    statuses = defaultdict(int)
    for name, _, _ in event_types:
        statuses[name] = 0
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        for name, where, regexp in event_types:
            if pairs.get("where_name") == where:
                match = regexp.search(pairs.get("message", ""))
                if match is not None:
                    statuses[name] += 1
                    break
    return statuses


def get_connection_reject_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    init_re = re.compile(r"connect from")
    rbl_re = re.compile(r"reject: CONNECT from(?:.*blocked by spam statistics)", flags=re.IGNORECASE)
    ratesrv_re = re.compile(r"reject by RateSrv: CONNECT")
    connection_manager_re = re.compile(r"reject by connection manager: CONNECT")
    event_types = (("new", "RECV", init_re),
                   ("reject_rbl", "RECV", rbl_re),
                   ("reject_ratesrv", "RECV", ratesrv_re),
                   ("reject_connection_manager", "RECV", connection_manager_re))
    return _get_reject_statuses(event_types, app, time, get_logs)


def get_sender_reject_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    init_re = re.compile(r"^from=")
    ratesrv_re = re.compile(r"reject from=.* by RateSrv")
    event_types = (("new", "RECV", init_re), ("reject_ratesrv", "RECV", ratesrv_re))
    return _get_reject_statuses(event_types, app, time, get_logs)


def get_recipient_reject_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    init_re = re.compile(r"^[\S]*$")
    ratesrv_re = re.compile(r"reject .* by RateSrv")
    rcsrv_re = re.compile(r"has exceeded message rate limit from")
    event_types = (("new", "RCPT", init_re), ("reject_ratesrv", "RCPT", ratesrv_re),
                   ("reject_rcsrv", "RC", rcsrv_re))
    return _get_reject_statuses(event_types, app, time, get_logs)


def get_bb_auth_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    auth_statuses = defaultdict(lambda: defaultdict(int))
    status_re = re.compile(r"method='(?P<method>\w+).*status='(?P<status>.*)'")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "BB-AUTH":
            match = status_re.search(pairs.get("message", ""))
            if match is not None:
                method = match.group("method").upper()
                status = match.group("status").split(':', 1)[0].lower()
                auth_statuses[method][status] += 1
    return auth_statuses


def get_hdr_from_stat(time=60, get_logs=timetail, app=NWSMTP_APP):
    stat = {"statuses": defaultdict(int), "errors": defaultdict(int)}
    hdr_from_re = re.compile(r"status=(?P<status>\w+).*error=\'(?P<error>.*)\'")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "HDR-FROM":
            match = hdr_from_re.search(pairs.get("message", ""))
            if match is not None:
                stat["statuses"][match.group("status")] += 1
                # Check for non-empty error:
                if match.group("error"):
                    stat["errors"][match.group("error")] += 1
    return stat


def get_settings_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    settings_statuses = defaultdict(lambda: defaultdict(int))
    status_re = re.compile(r"stat=(?P<status>\w+)( \((?P<reason>.*)\))?")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "SETTINGS":
            match = status_re.search(pairs.get("message", ""))
            if match is not None:
                settings_statuses[match.group('status')][match.group('reason')] += 1
    return settings_statuses


def get_settings_authorization_statuses(time=60, get_logs=timetail, app=NWSMTP_APP):
    bypass_by_bb_re = re.compile("stat=bypassed by BB status")
    bypass_by_options_re = re.compile("stat=bypassed by options")
    bypass_by_settings_error_re = re.compile("stat=bypassed by Settings error")
    reject_by_pop_and_imap_off_re = re.compile("stat=rejected by pop and imap off")
    reject_by_plain_auth_off_re = re.compile("stat=rejected by plain auth off")
    accept_re = re.compile("stat=accepted")
    event_types = (("bypass_by_bb", bypass_by_bb_re),
                   ("bypass_by_options", bypass_by_options_re),
                   ("bypass_by_settings_error", bypass_by_settings_error_re),
                   ("reject_by_pop_and_imap_off", reject_by_pop_and_imap_off_re),
                   ("reject_by_plain_auth_off", reject_by_plain_auth_off_re),
                   ("accept", accept_re))
    statuses = {}
    for name, _ in event_types:
        statuses[name] = 0
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        for name, regexp in event_types:
            if pairs.get("where_name") == "SETTINGS_AUTHORIZATION":
                match = regexp.search(pairs.get("message", ""))
                if match is not None:
                    statuses[name] += 1
                    break
    return statuses


def get_smtp_codes(time=60, get_logs=timetail, app=NWSMTP_APP):
    codes = defaultdict(int)
    reply_code_re = re.compile(r"(?P<code>\d+) .*")
    for pairs in get_logs(log_file=NWSMTP_LOGS[app], fmt=TIMETAIL_FMT, time=time):
        if pairs.get("where_name") == "REPLY":
            match = reply_code_re.search(pairs.get("message"))
            if match:
                codes[match.group("code")] += 1
    return codes
