import json
import logging
import os
import requests
import sys
import time
from mail.monitoring.common import conductor, qloud, deploy

DEFAULT_TTL = 20  # seconds

DEFAULT_YASM_AGENT_URL = "http://localhost:11005"
DEFAULT_YASM_AGENT_TIMEOUT = 2
YASM_AGENT_LOG_FILE = "/var/log/yandex/yasm-sender.log"
YASM_AGENT_DISABLED_FILE = "/app/monitoring/yasm/yasmagent.disabled"

HISTOGRAM_SUFFIXES = ["ahhh", "hhhh"]

UNISTAT_DIR = "/tmp/unistat/"
UNISTAT_TMP_DIR = UNISTAT_DIR + "tmp/"

log = logging.getLogger("yasm")


def get_project():
    if qloud.is_qloud():
        return qloud.get_component_path()
    return conductor.get_group()


def get_dc():
    if qloud.is_qloud():
        return qloud.environ["datacenter"]
    if deploy.is_deploy():
        return deploy.environ["node_meta"]["dc"]
    return conductor.get_dc()


def setup_logging(script_name):
    handler = logging.FileHandler(YASM_AGENT_LOG_FILE)
    handler.setFormatter(logging.Formatter('%(asctime)s - %(levelname)s script: ' + script_name + ',  %(message)s'))
    log.addHandler(handler)
    log.setLevel(logging.INFO)


def create_unistat_dirs():
    for directory in [UNISTAT_DIR, UNISTAT_TMP_DIR]:
        if not os.path.exists(directory):
            os.makedirs(directory)


def clean_old_unistsat_files():
    unists_files = [os.path.join(UNISTAT_DIR, f) for f in os.listdir(UNISTAT_DIR) if os.path.isfile(os.path.join(UNISTAT_DIR, f))]
    for file in unists_files:
        time_since_modified = time.time() - os.path.getmtime(file)
        if time_since_modified > DEFAULT_TTL:
            os.remove(file)


def save_unistat_data(metrics, suffix, prefix, geo, ctype):
    create_unistat_dirs()
    clean_old_unistsat_files()

    if prefix == "":
        unistat_filename = metrics.keys()[0]
    else:
        unistat_filename = prefix

    if ctype:
        unistat_filename = unistat_filename + "_" + ctype

    unistat_file = UNISTAT_DIR + unistat_filename
    unistat_tmp_file = UNISTAT_TMP_DIR + unistat_filename

    unistat_data = []
    for metric_name, value in metrics.iteritems():
        if prefix:
            metric_name = prefix + "_" + metric_name
        metric_name = metric_name + "_" + suffix

        unistat_metric = "geo={};ctype={};{}".format(geo, ctype, metric_name)

        unistat_data.append([unistat_metric, value])

    with open(unistat_tmp_file, "w") as f:
        json.dump(unistat_data, f)
    os.rename(unistat_tmp_file, unistat_file)


def send_yasmagent_data(metrics, suffix, prefix, prj, itype, geo, ctype, debug, dry_run, yasm_agent_url, ttl):
    if not prj:
        prj = get_project()

    if suffix in HISTOGRAM_SUFFIXES:
        ttl = 5

    yasm_data = [
        {
            "tags": {
                "prj": prj,
                "itype": itype,
                "geo": geo
            },
            "ttl": ttl,
        }
    ]
    if ctype is not None:
        yasm_data[0]["tags"]["ctype"] = ctype
    values = []
    for name, value in metrics.iteritems():
        if prefix:
            name = prefix + "_" + name
        value = {
            "name": name + "_" + suffix,
            "val": value
        }
        values.append(value)

    yasm_data[0]["values"] = values
    yasm_data = json.dumps(yasm_data)

    setup_logging(os.path.basename(sys.argv[0]))

    if debug:
        log.setLevel(logging.DEBUG)

    log.debug("metrics: " + str(dict(metrics)))
    log.debug("yasm_data: " + str(yasm_data))

    if not dry_run:
        try:
            response = requests.post(yasm_agent_url, data=yasm_data, timeout=DEFAULT_YASM_AGENT_TIMEOUT)
            log.info("yasm_agent code: " + str(response.status_code))
            log.debug("yasm_agent response: " + response.text)
            return response
        except requests.exceptions.ConnectionError as e:
            log.info("yasm agent: " + str(e))


def send(metrics, suffix, prefix="", prj=None, itype=None, geo=None, ctype=None, debug=False, dry_run=False,
         yasm_agent_url=DEFAULT_YASM_AGENT_URL, ttl=DEFAULT_TTL, mode=None):

    """ Sending metrics to local yasm agent.
        metrics - dictionary with metrics. e.g.: {"active": 1557, "deferred": 8939, "hold": 0, "incomming": 3}
        suffix - suffix for tell yasm, how aggregate metrics. Doc: https://wiki.yandex-team.ru/golovan/aggregation-types/#algoritmyagregacii
        prefix - additional string that can be added to metric name. No required.
        Finaly metric name in yasm will look like:
            push-[prefix_]metric_name-suffix
            e.g.: push-mailq_deferred_tmmm
        prj - Conductor (group) or Qloud(app) project
        itype - application name
        geo - geo region (datacenter)
        ctype - client service type
        debug - logging debug info
        dry_run - only logging metrics without sending to yasm_agent
        example: https://yasm.yandex-team.ru/chart/itype=postfix;hosts=CON;prj=mail_forward;signals=push-mailq_deferred_tmmm/

    """

    if not geo:
        geo = get_dc()

    save_unistat_data(metrics, suffix, prefix, geo, ctype)

    if mode != "unistat" and not os.path.exists(YASM_AGENT_DISABLED_FILE):
        send_yasmagent_data(metrics, suffix, prefix, prj, itype, geo, ctype, debug, dry_run, yasm_agent_url, ttl)
