#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse

from datetime import datetime

from mail.monitoring.common.platform import get_code_ranges, get_web_server_info
from mail.monitoring.common import monrun

NOW = datetime.now()

DAY_START_HOUR = 8
DAY_END_HOUR = 22

DEFAULT_CRIT_PERCENT = 10

CODES_CRIT_PERC = {
    400: {
        "save": {"default": 2},
        "send_mail": {"default": 30, "night": 60},
        "store": {"default": 15}
    },
    500: {
        "save": {"default": 2, "night": 10},
        "send_mail": {"default": 2},
        "store": {"default": 2, "night": 5},
    }
}


def is_night():
    if NOW.hour in xrange(DAY_START_HOUR, DAY_END_HOUR):
        return False
    return True


def main(args):

    status = monrun.OK
    description = ""

    http_info = get_web_server_info(args.log_filename, time=args.interval)
    request_count = sum(http_info["_all"]["_all"]["http_codes"].itervalues())

    # Мониторинг таймингов:
    if args.long_query_time:
        long_requests_count = http_info["_all"]["timing_buckets"][float("inf")]
        if request_count:
            long_requests_perc = int(
                float(long_requests_count) / float(request_count) * 100)

            if long_requests_perc > args.percent_crit:
                status = monrun.CRITICAL
            description = "%d%% requests longer than %dms" % (
                long_requests_perc, args.long_query_time)

    # Мониторинг "плохих" статус-кодов:
    bad_code_range = None
    if args.code:
        bad_code_range = [args.code]
    if args.code_range:
        bad_code_range = get_code_ranges(args.code_range)

    if bad_code_range:
        description = "OK"
        for handle, handle_info in http_info["_all"].iteritems():
            bad_codes_count = 0
            code_key = bad_code_range[0] / 100 * 100
            crit_limit = CODES_CRIT_PERC.get(code_key, args.percent_crit)
            if isinstance(crit_limit, dict):
                night_limit = crit_limit.get(handle, {}).get("night")
                crit_limit = crit_limit.get(handle, {}).get("default", args.percent_crit)
                if is_night() and night_limit is not None:
                    crit_limit = night_limit

            for code, count in handle_info["http_codes"].iteritems():
                if int(code) in bad_code_range:
                    bad_codes_count += count

            request_count = sum(http_info[handle]["_all"]["http_codes"].itervalues())
            if request_count:
                bad_codes_perc = int(float(bad_codes_count) / float(request_count) * 100)
                if bad_codes_perc > crit_limit:
                    status = monrun.CRITICAL
                    description = "%d%% of bad codes for %s" % (
                        bad_codes_perc, handle)

    monrun.report(status, description)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Checking http codes and timings")
    parser.add_argument("--log-filename", required=True,
                        type=str, help="Path to log file")
    parser.add_argument("--log-format", required=True,
                        type=str, help="Log format e.g.: [tskv, json, common]")
    parser.add_argument("-t", "--time", dest="interval", type=int, default=60, required=False,
                        help="Time interval(seconds) to perform check for")
    parser.add_argument("-p", "--percent-crit", dest="percent_crit", type=int, default=DEFAULT_CRIT_PERCENT,
                        required=False, help="Bad codes percent")
    parser.add_argument("-l", "--long-query-time", dest="long_query_time", type=int,
                        required=False, help="Long query time (in ms)")

    parser.add_argument("--x-reason-to-ignore", dest="x_reason_to_ignore", nargs="+", default=(),
                        required=False, help="List of reasons that will be ignore")

    group = parser.add_mutually_exclusive_group(required=False)
    group.add_argument("--code", type=int, help="Specific code rate")
    group.add_argument("--code-range", type=str, help="Range of bad codes. e.g.: 499:504")

    args = parser.parse_args()

    main(args)
