#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
from collections import defaultdict
from mail.monitoring.common import monrun
from mail.monitoring.delivery.lib import nwsmtp
from mail.monitoring.common import cal

AUTH_METHODS = ("PLAIN", "LOGIN", "XOAUTH2")
THRESHOLD = 5  # To suppress false positives when traffic is low (e.g night/weekend)


def calc_total_and_success(bb_auth_statuses):
    total_and_success = defaultdict(lambda: {"total": 0, "success": 0})
    for method, statuses in bb_auth_statuses.iteritems():
        total_and_success[method]["success"] = statuses["success"]
        total_and_success[method]["total"] = sum(statuses.values())

    return total_and_success


def make_description(status, total_and_success):
    description = "SUC/TOT: " + ", ".join("{0}={1[success]}/{1[total]}".format(k, v)
                                          for k, v in total_and_success.iteritems())
    if status == monrun.CRITICAL:
        return "no success for auth method; " + description
    return description


def main(methods, interval, app, mute_on_holiday):
    if mute_on_holiday and cal.check_holiday():
        monrun.report(monrun.OK, "Muted because of holiday")
        return

    total_and_success = calc_total_and_success(nwsmtp.get_bb_auth_statuses(time=interval, app=app))

    status = monrun.OK
    for method in methods:
        if total_and_success[method]["success"] == 0 and \
                total_and_success[method]["total"] > THRESHOLD:
            status = monrun.CRITICAL

    monrun.report(status, make_description(status, total_and_success))


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=("Checks that there were successful authorizations "
                     "for chosen types for last N minutes"))
    parser.add_argument("-t", dest="interval", type=int, default=60, required=False,
                        help="Time interval(seconds) to perform check for")
    parser.add_argument("-a", dest="auth_methods", choices=AUTH_METHODS, nargs="+", required=True,
                        help="List of auth methods, delimited with space")
    parser.add_argument("--app", dest="app", type=str, default=nwsmtp.NWSMTP_APP, required=False,
                        help="application type: [nwsmtp|nwsmtp-out] (default:nwsmtp)")
    parser.add_argument("--mute-on-holiday", action="store_true", required=False,
                        help="Mute check on holidays and weekends")
    args = parser.parse_args()
    auth_methods = map(str.upper, args.auth_methods)
    main(auth_methods, args.interval, args.app, args.mute_on_holiday)
