#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
from collections import defaultdict

from mail.monitoring.common import monrun
from mail.monitoring.delivery.lib import nwsmtp


DEFAULT_CRIT_PERCENT = 10.0
DEFAULT_YANDEX_TLD_FILE = "/etc/opendkim/opendkim-keylist"
DEFAULT_PDD_DOMAINS_FILE = "/var/yamail-dkim-pdd/pdd/opendkim-pdd-domains"


def get_tlds(tld_file):
    with open(tld_file, "r") as f:
        return [i.split()[0].split("_domainkey.")[1] for i in f.read().splitlines()]


def load_pdd_domains_with_keys(pdd_domains_file):
    """ Load pdd domains for which we have dkim keys."""
    with open(pdd_domains_file) as fd:
        return set(line.rstrip() for line in fd)


def main(interval, tld_file, skip_pdd_without_keys, pdd_domains_file, app):
    status = monrun.OK
    yandex_tlds = get_tlds(tld_file)
    dkim_statuses = nwsmtp.get_dkim_sign_statuses(time=interval, app=app)

    pdd_domains_with_keys = None
    if skip_pdd_without_keys:
        pdd_domains_with_keys = load_pdd_domains_with_keys(pdd_domains_file)

    result_counts = defaultdict(int)

    for domain, statuses_counts in dkim_statuses.iteritems():
        if domain in yandex_tlds and statuses_counts["miss"]:
            result_counts["yandex_missed"] += statuses_counts["miss"]
            continue

        if statuses_counts["ok"]:
            result_counts["ok_domains"] += 1

        if statuses_counts["miss"]:
            if skip_pdd_without_keys:
                if domain not in pdd_domains_with_keys:
                    result_counts["missed_domains_without_keys"] += 1
                    continue
            result_counts["missed_domains"] += 1

    total_missed = result_counts["missed_domains"] + result_counts["missed_domains_without_keys"]
    total = result_counts["ok_domains"] + total_missed
    if total:
        if result_counts["yandex_missed"] > 0:
            status = monrun.CRITICAL
        elif result_counts["missed_domains"] > 0:
            # unable to sign some domains but we have key files for them
            status = monrun.CRITICAL
    else:
        status = monrun.WARNING

    monrun.report(status, result_counts)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Checks nwsmtp dkim signing")
    parser.add_argument("-t", "--time", dest="interval", type=int,
                        default=60, help="Time interval(seconds) to perform check for")
    parser.add_argument("-f", "--tld-file", dest="tld_file", type=str,
                        default=DEFAULT_YANDEX_TLD_FILE, help="File with yandex TLD, one per line")
    parser.add_argument("--skip-pdd-without-keys", dest="skip_pdd_without_keys", action="store_true",
                        help="Do not count missed pdd domains when there are no keys for them in: %s" % DEFAULT_PDD_DOMAINS_FILE)
    parser.add_argument("--pdd-domains-file", dest="pdd_domains_file", type=str,
                        default=DEFAULT_PDD_DOMAINS_FILE, help="File with pdd domains, one per line")
    parser.add_argument("--app", dest="app", type=str, default=nwsmtp.NWSMTP_APP, required=False,
                        help="application type: [nwsmtp|nwsmtp-out] (default:nwsmtp)")
    args = parser.parse_args()
    main(**dict(args._get_kwargs()))
