#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
import glob
import re
import time
from mail.monitoring.common import monrun
from mail.monitoring.delivery.lib.common import check_port

ESET_PORT = 1344
ESET_LICENSE_PATH = "/opt/eset/esets/etc/license/*.lic"
ESET_UPDATE_DB_FILE = "/var/opt/eset/esets/lib/data/data.txt"

CRIT_LICENSE_DAYS = 14
WARN_LICENSE_DAYS = 21

CRIT_UPDATE_DB_HOURS = 48
WARN_UPDATE_DB_HOURS = 24

current_timestamp = int(time.time())


def check_alive():
    status = monrun.OK
    description = "OK"

    if check_port(ESET_PORT) != 1:
        status = monrun.CRITICAL
        description = "eset's port is down"
    monrun.report(status, description)


def get_database_update_ts():
    last_update_re = re.compile(r"LastUpdate=(?P<last_update_ts>\d+)")
    with open(ESET_UPDATE_DB_FILE) as f:
        for line in f:
            match = last_update_re.search(line)
            if match:
                last_update_ts = int(match.group("last_update_ts"))
                return float(current_timestamp - last_update_ts)


def check_database():
    status = monrun.OK

    if not glob.glob(ESET_UPDATE_DB_FILE):
        status = monrun.CRITICAL
        monrun.report(status, "Can't find file: " + ESET_UPDATE_DB_FILE)
        return 0

    hours_updated = float(get_database_update_ts() / 3600)
    if hours_updated > WARN_UPDATE_DB_HOURS:
        status = monrun.WARNING
    if hours_updated > CRIT_UPDATE_DB_HOURS:
        status = monrun.CRITICAL
    monrun.report(status, "db updated " +
                  str(round(hours_updated, 1)) + " hrs ago")


def iter_license_expire_date():
    expire_re = re.compile(
        r"NODE NAME=\"EXPIRE\".*VALUE=\"(?P<expire_date>\w+)")

    for license_file in glob.glob(ESET_LICENSE_PATH):
        with open(license_file) as f:
            for line in f:
                match = expire_re.search(line)
                if match:
                    hex_date = match.group("expire_date")
                    yield int(hex_date, 16)


def check_license():
    status = monrun.OK

    if not glob.glob(ESET_LICENSE_PATH):
        status = monrun.CRITICAL
        monrun.report(status, "Can't find license: " + ESET_LICENSE_PATH)
        return 0

    for expire_date in iter_license_expire_date():
        days_before_expire = (expire_date - current_timestamp) / 86400
        if days_before_expire < WARN_LICENSE_DAYS:
            status = monrun.WARNING
            break
        if days_before_expire < CRIT_LICENSE_DAYS:
            status = monrun.CRITICAL
            break

    monrun.report(status, str(days_before_expire) +
                  " days before license expire")


parser = argparse.ArgumentParser()
subparsers = parser.add_subparsers()
alive_parser = subparsers.add_parser("alive")
alive_parser.set_defaults(func=check_alive)

database_parser = subparsers.add_parser("database")
database_parser.set_defaults(func=check_database)

database_parser = subparsers.add_parser("license")
database_parser.set_defaults(func=check_license)

if __name__ == "__main__":
    args = parser.parse_args()
    args.func()
