#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
import socket

from mail.monitoring.common import monrun, util
from mail.monitoring.delivery.lib import nwsmtp


DEFAULT_LIMITS = {"crit": "10%", "warn": "5%", "not-found": "40%"}
DEFAULT_DISCARD_LIST_FILE = "/etc/nwsmtp/rcpt_discard_list"


def _check(limit, status_count, total):
    if limit["is_percent"]:
        compare_val = float(status_count) / total * 100
    else:
        compare_val = status_count
    return compare_val > limit["value"]


def load_discard_list(discard_list_file):
    with open(discard_list_file) as fd:
        return set(line.rstrip() for line in fd)


def main(crit_limit, warn_limit, not_found_limit, discard_list_file, app):
    status = monrun.OK

    filtered_rcpts = {"root@" + socket.getfqdn()}
    if discard_list_file:
        filtered_rcpts.update(load_discard_list(discard_list_file))

    ml_statuses = nwsmtp.get_ml_statuses(filtered_rcpts=filtered_rcpts, app=app)
    ml_statuses_sum = sum([i for i in ml_statuses.values()])

    if ml_statuses_sum:
        crit_limit = util.transform_param(crit_limit)
        warn_limit = util.transform_param(warn_limit)
        not_found_limit = util.transform_param(not_found_limit)

        if _check(warn_limit, ml_statuses['error'], ml_statuses_sum):
            status = monrun.WARNING
        if _check(crit_limit, ml_statuses['error'], ml_statuses_sum) or _check(
                not_found_limit, ml_statuses['not_found'], ml_statuses_sum):
            status = monrun.CRITICAL

    monrun.report(status, ml_statuses)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=("Checks ML responses"))
    parser.add_argument("-c", "--crit", type=str, dest="crit_limit",
                        default=DEFAULT_LIMITS["crit"], help="Critical errors limit")
    parser.add_argument("-w", "--warn", type=str, dest="warn_limit",
                        default=DEFAULT_LIMITS["warn"], help="Warning errors limit")
    parser.add_argument("-n", "--not-found", type=str, dest="not_found_limit",
                        default=DEFAULT_LIMITS["not-found"], help="Critical not-found error limit")
    parser.add_argument("-d", "--discard-list", type=str, dest="discard_list_file",
                        default=DEFAULT_DISCARD_LIST_FILE,
                        help="File with emails to discard, one per line (default={}".format(DEFAULT_DISCARD_LIST_FILE))
    parser.add_argument("--app", dest="app", type=str, default=nwsmtp.NWSMTP_APP, required=False,
                        help="application type: [nwsmtp|nwsmtp-out] (default:nwsmtp)")
    args = parser.parse_args()
    main(**dict(args._get_kwargs()))
