#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
from mail.monitoring.common import monrun, util
from mail.monitoring.delivery.lib import nwsmtp

DEFAULT_LIMITS = {"accept": "1%", "spam": "1", "error": "10%", "reject": "1"}


def _check(limit, status_count, total, check_min=True):
    if limit["is_percent"]:
        compare_val = float(status_count) / total * 100
    else:
        compare_val = status_count

    # check_min true if we check minimum (all types exclude errors)
    if check_min:
        if compare_val < limit["value"]:
            return True
    else:
        if compare_val > limit["value"]:
            return True

    return False


def main(args):
    status = monrun.OK

    so_statuses = nwsmtp.get_so_statuses(time=args.interval, app=args.app)
    so_statuses_sum = sum([i for i in so_statuses.values()])

    if so_statuses_sum:
        accept_limit = util.transform_param(args.accept)
        spam_limit = util.transform_param(args.spam)
        reject_limit = util.transform_param(args.reject)
        error_limit = util.transform_param(args.error)

        crit_reasons = []
        if _check(accept_limit, so_statuses["accept"], so_statuses_sum):
            crit_reasons.append("accept<{0}".format(args.accept))
        if _check(spam_limit, so_statuses["spam"], so_statuses_sum):
            crit_reasons.append("spam<{0}".format(args.spam))
        if _check(reject_limit, so_statuses["reject"], so_statuses_sum):
            crit_reasons.append("reject<{0}".format(args.reject))
        if _check(error_limit, so_statuses["error"], so_statuses_sum, False):
            crit_reasons.append("error>{0}".format(args.error))

        if len(crit_reasons) > 0:
            status = monrun.CRITICAL
            so_statuses = (crit_reasons, so_statuses)

    monrun.report(status, so_statuses)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=("Checks SO resolution states"))
    parser.add_argument("-x", "--accept", dest="accept", type=str,
                        default=DEFAULT_LIMITS["accept"], help="Minimum accept limit")
    parser.add_argument("-s", "--spam", dest="spam", type=str,
                        default=DEFAULT_LIMITS["spam"], help="Minimum spam limit")
    parser.add_argument("-m", "--reject", dest="reject", type=str,
                        default=DEFAULT_LIMITS["reject"], help="Minimum reject limit")
    parser.add_argument("-e", "--error", dest="error", type=str,
                        default=DEFAULT_LIMITS["error"], help="Maximum error limit")
    parser.add_argument("-t", "--time", dest="interval", type=int, default=60, required=False,
                        help="Time interval(seconds) to perform check for")
    parser.add_argument("--app", dest="app", type=str, default=nwsmtp.NWSMTP_APP, required=False,
                        help="application type: [nwsmtp|nwsmtp-out] (default:nwsmtp)")
    main(parser.parse_args())
