#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
import operator
import re
import time
from mail.monitoring.common import monrun
from mail.monitoring.delivery.lib.common import get_delivery_statuses
from mail.monitoring.delivery.lib.postfix import get_queue_size, get_queue_data


POSTFIX_QUEUES = ["active", "autohold", "deferred", "hold", "incoming", "mailer_daemon"]
POSTFIX_STATUSES = ["sent", "bounced", "deferred", "expire"]
LONGRUN_TIME = 120
LONGRUN_WARN_PERCENT = 20
LONGRUN_CRIT_PERCENT = 40


def check_queue_size(args):
    status = monrun.OK
    queue_size = get_queue_size(args.queue)

    if queue_size > args.crit_limit:
        status = monrun.CRITICAL
    monrun.report(status, queue_size)


def check_statuses(args):
    status = monrun.OK
    statuses_count = get_delivery_statuses(app="postfix", time=args.interval)[args.status]
    if args.status == 'sent':
        if statuses_count < args.crit_limit:
            status = monrun.CRITICAL
    else:
        if statuses_count > args.crit_limit:
            status = monrun.CRITICAL
    monrun.report(status, statuses_count)


def _check_recipients(recipients, excludes_re):
    if not excludes_re:
        return True

    for recipient in recipients:
        if not reduce(operator.or_, [bool(e.search(recipient["address"])) for e in excludes_re], False):
            return True
    return False


def check_longruns(args):
    status = monrun.OK
    start_longrun = int(time.time()) - args.time * 60
    exclude_recipients = [re.compile(item.strip()) for item in args.exclude_rcpts.split(',')] if args.exclude_rcpts else []
    queue = filter(lambda q: _check_recipients(q["recipients"], exclude_recipients), get_queue_data(args.queue_name))
    longrun_queue = filter(lambda q: int(q["arrival_time"]) < start_longrun, queue)
    total = len(queue)
    longruns = len(longrun_queue)
    percent = int(float(longruns) / total * 100) if total else 0
    desc = "{}/{} ({}%)".format(longruns, total, percent)
    if longruns > 0:
        if (args.crit_limit and longruns >= args.crit_limit) or percent >= args.percent_crit:
            status = monrun.CRITICAL
        elif (args.warn_limit and longruns >= args.warn_limit) or percent >= args.percent_warn:
            status = monrun.WARNING
    monrun.report(status, desc)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers()

    queue_parser = subparsers.add_parser("queue")
    queue_parser.add_argument(
        dest="queue", choices=POSTFIX_QUEUES, action="store")
    queue_parser.add_argument("-l", "--crit-limit",
                              action="store", type=int, default=0)
    queue_parser.set_defaults(func=check_queue_size)

    status_parser = subparsers.add_parser("status")
    status_parser.add_argument(
        dest="status", choices=POSTFIX_STATUSES, action="store")
    status_parser.add_argument(
        "-l", "--crit-limit", action="store", type=int, default=0)
    status_parser.add_argument(
        "-t", "--interval", action="store", type=int, default=60, help="Time interval(seconds) to perform check for")
    status_parser.set_defaults(func=check_statuses)

    longrun_parser = subparsers.add_parser("longrun")
    longrun_parser.add_argument(
        "-q", "--queue", dest="queue_name", choices=POSTFIX_QUEUES, default="deferred",
        help="Postfix queue name to check")
    longrun_parser.add_argument(
        "-t", "--time", dest="time", type=int, default=LONGRUN_TIME,
        help="Minimum time for queue element to treat it as long-run, minutes")
    longrun_parser.add_argument(
        "-c", "--percent-crit", dest="percent_crit", type=int, default=LONGRUN_CRIT_PERCENT,
        help="Critical percent of longrun mails")
    longrun_parser.add_argument(
        "-w", "--percent-warn", dest="percent_warn", type=int, default=LONGRUN_WARN_PERCENT,
        help="Warning percent of longrun mails")
    longrun_parser.add_argument(
        "-C", "--crit-limit", dest="crit_limit", type=int,
        help="Critical amount of longrun mails")
    longrun_parser.add_argument(
        "-W", "--warn-limit", dest="warn_limit", type=int,
        help="Warning amount of longrun mails")
    longrun_parser.add_argument(
        "-e", "--exclude-rcpts", dest="exclude_rcpts", type=str, default="",
        help="Exclude recepients regex separated by comma")
    longrun_parser.set_defaults(func=check_longruns)

    args = parser.parse_args()
    args.func(args)
