#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
import re
import time
import json
import os
from collections import namedtuple
from mail.monitoring.common import monrun
from mail.monitoring.delivery.lib.postfix import get_queue_data
from mail.monitoring.common.timetail import timetail

NSLS_LOG = "/var/log/notsolitesrv/notsolitesrv.tskv"


COMMON_ERRORS = {"db_init_error": "user is not initialized in the database",
                 "mulcagate_error": "mulcagate error",
                 "sharpei_error": "sharpei error",
                 "store_db_error": "store to db error",
                 "unique_violation": "unique_violation"}

NSLS_ERROR_MESSAGE = "Sorry, service unavailable;"
NSLS_ERRORS_CACHE_FILE = "/tmp/nsls_errors.json"
NSLS_ERRORS_CACHE_TTL = 300  # Seconds

LONGRUN_TIME = 120  # Minutes
LONGRUN_QUEUE_SIZE_LIMIT = 30
POSTFIX_MAXIMAL_BACKOFF_TIME = 400  # Seconds

NSLS_Error = namedtuple("NSLS_Error", "email type session")

COMMON_ERRORS_RE = COMMON_ERRORS.copy()
for error_type, error_text in COMMON_ERRORS_RE.iteritems():
    COMMON_ERRORS_RE[error_type] = re.compile(error_text)


def get_longrun_queue(args):
    start_longrun = int(time.time()) - args.longrun_time * 60
    longrun_queue = filter(lambda q: int(
        q["arrival_time"]) < start_longrun, get_queue_data())
    return longrun_queue


def check_lognrun_queue_size(args, longrun_queue):
    if len(longrun_queue) > args.longrun_queue_size_limit:
        return False
    return True


def get_nsls_error_sessions_from_longrun_queue(args, longrun_queue):
    error_sessions = {}
    for letter in longrun_queue:
        for recipient_data in letter["recipients"]:
            delay_reason = recipient_data.get("delay_reason", "")
            if re.search(NSLS_ERROR_MESSAGE, delay_reason):
                error_session = delay_reason.split(
                    NSLS_ERROR_MESSAGE)[1].split()[0]
                error_sessions[error_session] = recipient_data["address"]
    return error_sessions


def find_common_error_in_log(nsls_session, log):
    re_session = re.compile(nsls_session)
    for line in log:
        if not all(key in line for key in ("uniq_id", "message")):
            continue
        if re_session.search(line["uniq_id"]):
            for error_type, error_re in COMMON_ERRORS_RE.iteritems():
                if error_re.search(line["message"]):
                    return error_type
    return "unknown"


def save_errors_to_cache(errors):
    with open(NSLS_ERRORS_CACHE_FILE, "w") as f:
        cache_data = {"timestamp": int(time.time()), "data": errors}
        json_cache_data = json.dumps(cache_data)
        f.write(json_cache_data)


def get_nsls_errors_from_queue(args, longrun_queue):
    if (os.path.isfile(NSLS_ERRORS_CACHE_FILE)
            and not args.no_cache
            and not args.error_type == "unknown"):
        with open(NSLS_ERRORS_CACHE_FILE) as f:
            cache_data = json.load(f)
        if int(time.time()) - cache_data["timestamp"] < NSLS_ERRORS_CACHE_TTL:
            return [NSLS_Error(*i) for i in cache_data["data"]]

    error_sessions = get_nsls_error_sessions_from_longrun_queue(
        args, longrun_queue)

    log = timetail(NSLS_LOG, fmt="tskv", time=args.postfix_maximal_backoff_time)

    nsls_errors = [NSLS_Error(
        session=nsls_session,
        email=recipient,
        type=find_common_error_in_log(nsls_session, log))
        for nsls_session, recipient in error_sessions.iteritems()]

    save_errors_to_cache(nsls_errors)
    return nsls_errors


def check_errors_by_type(args, longrun_queue):
    status = monrun.OK
    nsls_errors = get_nsls_errors_from_queue(args, longrun_queue)
    nsls_type_errors = filter(
        lambda session: session.type == args.error_type, nsls_errors)
    if len(nsls_type_errors) > args.errors_crit_count:
        status = monrun.CRITICAL

    unique_errors = {error.email: error for error in nsls_type_errors}

    if args.hide_session_id:
        description = unique_errors.keys()
    else:
        description = ["{0.email}: {0.session}".format(error) for error in unique_errors.itervalues()]
    monrun.report(status, description)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=("Checking emails that are long in the queue"))
    parser.add_argument("-c", "--errors-crit-count", type=int,
                        required=False, default=0)
    parser.add_argument("--error-type", type=str, default="unknown", required=False,
                        help="Type of error: " + str(COMMON_ERRORS.keys()))
    parser.add_argument("--no-cache", action="store_true",
                        help="Disable cache")
    parser.add_argument("-t", "--longrun-time", type=int, required=False,
                        help="Time the letter spent in the queue (minutes)", default=LONGRUN_TIME)
    parser.add_argument("--postfix-maximal-backoff-time", type=int,
                        required=False, default=POSTFIX_MAXIMAL_BACKOFF_TIME)
    parser.add_argument("--longrun_queue_size_limit", type=int,
                        required=False, default=LONGRUN_QUEUE_SIZE_LIMIT)
    parser.add_argument("--hide_session_id", action="store_true",
                        required=False, help="Do not show session_id in description")

    args = parser.parse_args()

    longrun_queue = get_longrun_queue(args)

    # В случае, если очередь longrun переполнена не поджигаем проверки с
    # аргументом "error_type":
    if not check_lognrun_queue_size(args, longrun_queue):
        status = monrun.WARNING if args.error_type else monrun.CRITICAL
        monrun.report(status, "Longrun queue overflow")
        exit()

    check_errors_by_type(args, longrun_queue)
