#!/usr/local/bin/python
# -*- coding: UTF-8 -*-

import json
import os
import requests
from os.path import expanduser

QLOUD_TOKEN_FILE = expanduser("~") + "/.config/qloud_token"

QLOUD_API_URL = "https://qloud-ext.yandex-team.ru/api/v1/environment"
QLOUD_DUMP_URL = QLOUD_API_URL + "/dump/{env_name}"
QLOUD_UPDATE_URL = QLOUD_API_URL + "/upload/return-header?targetState=DEPLOYED&skipIfNoChange=false"
QLOUD_CURRENT_VERSION_URL = QLOUD_API_URL + "/current/{env_name}"
QLOUD_FAST_DEPLOY_URL = QLOUD_API_URL + "/fast-deploy/{env_name}/{version}"

_qloud_token = None


def get_qloud_token():
    global _qloud_token

    if _qloud_token:
        return _qloud_token

    if "QLOUD_TOKEN" in os.environ:
        _qloud_token = os.environ["QLOUD_TOKEN"]
    else:
        with open(QLOUD_TOKEN_FILE) as f:
            _qloud_token = f.read().strip()
    return _qloud_token


def get_oauth_header():
    return {"Authorization": "OAuth " + get_qloud_token()}


def check_http_ok(resp, message):
    if resp.status_code / 100 != 2:
        raise RuntimeError("Bad response code %s on '%s'", resp.status_code, message)


def get_qloud_environment_dump(name):
    resp = requests.get(QLOUD_DUMP_URL.format(env_name=name), headers=get_oauth_header())
    check_http_ok(resp, "environment dump")
    return resp.json()


def qloud_upload_environment(env):
    headers = get_oauth_header()
    headers["content-type"] = "application/json"
    resp = requests.post(QLOUD_UPDATE_URL, headers=headers, data=json.dumps(env))
    check_http_ok(resp, "upload environment")
    return resp.json()


def get_qloud_current_version_info(name):
    resp = requests.get(QLOUD_CURRENT_VERSION_URL.format(env_name=name),
                        headers=get_oauth_header())
    check_http_ok(resp, "current_version_info")
    return resp.json()


def qloud_fast_deploy(name, version, deploy_info):
    headers = get_oauth_header()
    headers["content-type"] = "application/json"
    resp = requests.post(QLOUD_FAST_DEPLOY_URL.format(env_name=name, version=version),
                         headers=headers,
                         data=json.dumps(deploy_info))
    check_http_ok(resp, "fast deploy")
