#!/usr/local/bin/python
# -*- coding: UTF-8 -*-

import argparse
import calendar
import copy
import deepdiff
import json
import re
import requests
from datetime import datetime, timedelta
from urllib import urlencode

from ..qloud import get_qloud_environment_dump, qloud_upload_environment

DEFAULT_VERSION_LIMIT = 10
SANDBOX_REOURCE_API = "https://sandbox.yandex-team.ru/api/v1.0/resource?"


def deploy_resource(args):
    environment = get_qloud_environment_dump(args.environment_name)
    if not environment:
        return
    environment_orig = copy.deepcopy(environment)

    environment["comment"] = "update monitorings: sandbox resource id %s" % args.resource_version
    for component in environment["components"]:
        component_name = component["componentName"]
        for resource in component["sandboxResources"]:
            if re.search(args.resource_name, resource["localName"]):
                resource["id"] = args.resource_version
                print "resource %s will be updated for %s " % (args.resource_name, component_name)
    diffs = deepdiff.DeepDiff(environment_orig, environment)

    if diffs.get("values_changed"):
        for item, changes in diffs.get("values_changed").iteritems():
            print "%s\n%s\n\n" % (item, changes)

    if args.dry_run is False:
        qloud_upload_environment(environment)


def utc_to_local(utc_dt):
    timestamp = calendar.timegm(utc_dt.timetuple())
    local_dt = datetime.fromtimestamp(timestamp)
    assert utc_dt.resolution >= timedelta(microseconds=1)
    return local_dt.replace(microsecond=utc_dt.microsecond)


def get_resource_info(resource_name, version_limit):
    post_data = {"limit": version_limit,
                 "offset": 0,
                 "attrs": json.dumps({"resource_name": resource_name})}
    resource_info = requests.get(
        SANDBOX_REOURCE_API + urlencode(post_data)).json()["items"]
    return resource_info


def print_resource(resource_info):
    for resource in resource_info:
        created_time = utc_to_local(datetime.strptime(
            resource["time"]["created"], '%Y-%m-%dT%H:%M:%SZ'))
        print "%d\t%s\t%s" % (resource["id"], created_time, resource["file_name"],)


def view_resource(args):
    resource_info = get_resource_info(args.resource_name, args.limit)
    print_resource(resource_info)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=("Sandbox and Qloud resource manager"))
    subparsers = parser.add_subparsers()
    view_parser = subparsers.add_parser("view")
    view_parser.add_argument("-n", "--resource-name",
                             type=str, required=True, help="Resource name")
    view_parser.add_argument("-l", "--limit", type=int, default=DEFAULT_VERSION_LIMIT,
                             required=False, help="API result limit")
    view_parser.set_defaults(func=view_resource)

    deploy_parser = subparsers.add_parser("deploy")
    deploy_parser.add_argument("-e", "--environment-name", type=str,
                               required=True, help="Qloud environment name. f.e.: mail.mdbsave.production")
    deploy_parser.add_argument("-r", "--resource-version", type=int, default=0,
                               required=True, help="Resource version")
    deploy_parser.add_argument(
        "-n", "--resource-name", type=str, required=True, help="Resource name")
    deploy_parser.add_argument("-d", "--dry_run", action='store_true', default=False,
                               required=False, help="Dry run mode")
    deploy_parser.set_defaults(func=deploy_resource)

    args = parser.parse_args()
    args.func(args)
