# -*- coding: UTF-8 -*-

import argparse
import logging
import sys

root = logging.getLogger()
root.setLevel(logging.DEBUG)
root.addHandler(logging.StreamHandler(sys.stdout))

from ..qloud import get_qloud_current_version_info, qloud_fast_deploy, get_qloud_environment_dump

COMPONENTS = {
    "mail.nwsmtp.qa": {
        "mxback",
        "mxfront",
        "yaback"
    }
}


def build_component_environment_for_update(comp_name, comp_env):
    return {
        "componentName": comp_name,
        "properties": {
            "componentEnvironment": build_component_environment(comp_env)
        }
    }


def get_component_environment(comp_env):
    return dict(item.split("=", 1) for item in comp_env.split("\n"))


def build_component_environment(comp_env):
    return "\n".join("%s=%s" % (k, v) for k, v in comp_env.iteritems())


def get_packages(packages):
    return dict(item.split("=", 1) for item in packages.split(" ") if "=" in item)


def build_packages(packages):
    items = sorted(packages.items(), key=lambda item: item[0])
    return " ".join("%s=%s" % (k, v) for k, v in items)


def find_component(env_info, name):
    components = env_info["components"]

    comp = next((comp for comp in components
                 if comp["componentName"] == name), None)
    if not comp:
        raise RuntimeError(
            "Component %s not found in %s " % (name, ",".join(c["componentName"] for c in components))
        )
    return comp


def update_package_version(packages, pkg_name, comp_name, version):
    pkg_name = pkg_name.replace("_", "-")
    config = pkg_name + "-" + comp_name
    if config in packages:
        packages[config] = version
        return
    if pkg_name in packages:
        packages[pkg_name] = version


def main(pkg_versions):
    for env_name, components in COMPONENTS.iteritems():
        current_version = get_qloud_current_version_info(env_name)["version"]
        update_context = []

        env_info = get_qloud_environment_dump(env_name)
        for comp_name in components:
            comp = find_component(env_info, comp_name)
            comp_env = get_component_environment(comp["properties"]["componentEnvironment"])

            packages = get_packages(comp_env["PACKAGES"])
            for pkg_name, version in pkg_versions.iteritems():
                update_package_version(packages, pkg_name, comp_name, version)
            comp_env["PACKAGES"] = build_packages(packages)

            update_context.append(
                build_component_environment_for_update(comp_name, comp_env)
            )
        qloud_fast_deploy(env_name, current_version, update_context)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Update packages for nwsmtp(-corp)/qa")
    parser.add_argument("--nwsmtp", type=str)
    parser.add_argument("--nwsmtp-cfg", type=str)
    parser.add_argument("--smtpgate", type=str)
    parser.add_argument("--smtpgate-config", type=str)
    parser.add_argument("--notsolitesrv", type=str)
    parser.add_argument("--notsolitesrv-config", type=str)
    args = parser.parse_args()
    pkgs = {k: v for k, v in args._get_kwargs() if v}
    if not pkgs:
        raise RuntimeError("No versions was specified")
    main(pkgs)
