#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
import os
from ast import literal_eval
from datetime import datetime
from collections import defaultdict
from itertools import chain
from mail.monitoring.delivery.lib.postfix import get_queue_data, hold_ids

DEFAULT_RECIPIENT_LIMIT = 20
DEFAULT_QUEUES = ['deferred']

IDS_FILE = '/var/tmp/postfix_hold.lst'
UNHOLD_IDS_FILE = '/var/tmp/postfix_unhold.lst'
LOG_FILE = '/var/log/postfix_hold.log'
AMOUNT_FILE = IDS_FILE + '.amount'


def get_queue(queues):
    return chain(*map(get_queue_data, queues))


def get_freqs_and_ids(queue):
    recipients = defaultdict(int)
    recipients_queue = defaultdict(list)
    for msg in queue:
        for recipient in msg["recipients"]:
            recipients[recipient["address"]] += 1
            recipients_queue[recipient["address"]].append(msg["queue_id"])
    return recipients, recipients_queue


def get_bad_recipients_ids(rcpts_limit, recipients, recipients_queue):
    rcpts_ids = []
    for recipient, count in recipients.iteritems():
        if count >= rcpts_limit:
            rcpts_ids += map(lambda id: (recipient, id), recipients_queue[recipient])
    return rcpts_ids


def actualize_ids_file():
    hold_ids = [i["queue_id"] for i in get_queue_data("hold")]
    if not os.path.isfile(IDS_FILE):
        with open(IDS_FILE, 'w') as ids_file:
            ids_file.write("")
    with open(IDS_FILE, "r") as ids_file:
        ids_file_data = ids_file.read().splitlines()
        ids_file_data = [i for i in ids_file_data if i != ""]
    file_rcpts_ids = map(literal_eval, ids_file_data)
    file_ids = [(rcpt, queue_id) for rcpt, queue_id in file_rcpts_ids if queue_id in hold_ids]
    amount = len(set([queue_id for rcpt, queue_id in file_rcpts_ids]))
    file_ids = "\n".join(str(i) for i in file_ids) + "\n"
    with open(IDS_FILE, 'w') as ids_file:
        ids_file.writelines(file_ids)
    with open(AMOUNT_FILE, 'w') as amount_file:
        amount_file.write(str(amount))


def process_ids(rcpts_ids):
    actualize_ids_file()
    # during unholding, some emails may be considered as subjects
    # to autohold and to avoid holding them again, we should look into
    # unhold list
    rcpts_ids = [(rcpt, id) for rcpt, id in rcpts_ids]
    ids = map(lambda (_, id): id, rcpts_ids)
    if len(ids) == 0:
        return
    # convert them to string
    rcpts_ids = "\n".join(str(i) for i in rcpts_ids) + "\n"
    with open(IDS_FILE, 'a') as ids_file:
        ids_file.writelines(rcpts_ids)
    hold_ids(ids)
    with open(IDS_FILE, 'r') as ids_file:
        num_ids = sum(1 for line in ids_file)
    with open(AMOUNT_FILE, 'w') as amount_file:
        amount_file.write(str(num_ids) + '\n')
    with open(LOG_FILE, 'a') as log_file:
        curr_time = datetime.now().strftime('%Y.%m.%d %H:%M:%S')
        timemarked_ids = map(lambda i: curr_time + ": " + i, rcpts_ids)
        log_file.writelines(timemarked_ids)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Autoholder")
    parser.add_argument("-n", "--rcpts-limit",
                         type=int, default=DEFAULT_RECIPIENT_LIMIT,
                         help="Threshold")
    parser.add_argument("-q", "--queues",
                         choices=['active', 'incoming', 'deferred', 'hold'],
                         action='append', default=DEFAULT_QUEUES,
                         help="Queues")
    args = parser.parse_args()
    queue = get_queue(args.queues)
    rcpts, rcpts_ids = get_freqs_and_ids(queue)
    ids_to_hold = get_bad_recipients_ids(args.rcpts_limit, rcpts, rcpts_ids)
    process_ids(ids_to_hold)
