#!/usr/sbin/monpy
# -*- coding: UTF-8 -*-

import argparse
import os
import time
from ast import literal_eval
from collections import defaultdict
from mail.monitoring.delivery.lib.postfix import get_queue_data

DEFAULT_RECIPIENT_LIMIT = 100
DEFAULT_PAUSE_TIME = 250
DEFAULT_IDS_UPDATE_INTERVAL = 60000

IDS_FILE = '/var/tmp/postfix_hold.lst'
UNHOLD_FILE = '/var/tmp/postfix_unhold.lst'
UNHOLD_AMOUNT_FILE = UNHOLD_FILE + '.amount'


def get_ids_in_hold_queue():
    holded = [item['queue_id'] for item in get_queue_data('hold')]
    return set(holded)


def iter_by_rcpts(by_rcpt):
    while by_rcpt:
        ids = [q.pop() for q in by_rcpt.values()]
        yield ids
        by_rcpt = {k: v for k, v in by_rcpt.iteritems() if len(v) > 0}


def form_packs(ids_iter, recipient_limit):
    ids = []
    for block in ids_iter:
        for id in block:
            ids.append(id)
            if len(ids) >= recipient_limit:
                yield ids
                ids = []
        if ids:
            yield ids
            ids = []


def get_holded(processed_ids):
    try:
        with open(IDS_FILE, 'r') as ids_file:
            ids = ids_file.read().splitlines()
    except:
        return []
    # restore tuples (rcpt, id)
    rcpts_ids = map(lambda s: literal_eval(s), ids)
    rcpts_ids = [(rcpt, id) for rcpt, id in rcpts_ids if id not in processed_ids]
    return rcpts_ids


def factorize(all_holded, rcpts_ids, ids):
    by_rcpt = defaultdict(list)
    num = 0
    for rcpt, id in rcpts_ids:
        if id in all_holded:
            by_rcpt[rcpt].append(id)
            num += 1
        else:
            ids.add(id)
    return by_rcpt, num


def unhold_them_all(recipient_limit, pause_time, update_interval, debug=False):
    ids = set()
    rcpts_ids = get_holded(ids)
    all_holded = get_ids_in_hold_queue()
    by_rcpt, num = factorize(all_holded, rcpts_ids, ids)
    if debug:
        print("Total to process: %d" % num)
    while by_rcpt:
        time_to_update = update_interval
        for pack in form_packs(iter_by_rcpts(by_rcpt), recipient_limit):
            if debug:
                print("Unholding %d letters..." % (len(pack),))
            for queue_id in pack:
                ids.add(queue_id)
                os.system("postsuper -H " + queue_id)
            speedup_factor = 0.2 + (len(pack) * 0.8) / recipient_limit
            if debug:
                print("Speedup factor: %f" % speedup_factor)
            time.sleep((pause_time * speedup_factor) / 1000.0)
            time_to_update -= pause_time
            if time_to_update <= 0:
                if debug:
                    print("Updating from hold list...")
                break
        rcpts_ids = get_holded(ids)
        all_holded = get_ids_in_hold_queue()
        by_rcpt, num = factorize(all_holded, rcpts_ids, ids)
        if debug:
            print("Total to process after update: %d" % num)
        # truncating
        # lock due to autohold's need to look at unhold file
        with open(UNHOLD_FILE, 'w'):
            pass
        with open(UNHOLD_AMOUNT_FILE, 'w') as amount_file:
            amount_file.write('0\n')
    os.system("postfix flush")


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Unholder")
    parser.add_argument("-n", "--rcpts-limit",
                         type=int, default=DEFAULT_RECIPIENT_LIMIT,
                         help="Maximum number of defferent recipients in on pack")
    parser.add_argument("-t", "--pause-time",
                         type=int, default=DEFAULT_PAUSE_TIME,
                         help="Delay time between unholding chunks")
    parser.add_argument("-u", "--update-interval",
                         type=int, default=DEFAULT_IDS_UPDATE_INTERVAL,
                         help="Time between updates from new hold list")
    parser.add_argument("-v", "--verbose",
                         action='store_true', default=False,
                         help="Show some debug info")
    args = parser.parse_args()
    unhold_them_all(args.rcpts_limit, args.pause_time, args.update_interval, args.verbose)
