#!/usr/bin/python
# -*- coding: UTF-8 -*-

import argparse
import calendar
import copy
import deepdiff
import json
import re
import requests
from datetime import datetime, timedelta
from os.path import expanduser
from pprint import pprint
from urllib import urlencode


QLOUD_TOKEN_FILE = expanduser("~") + "/.config/qloud_token"

QLOUD_API_URL = "https://qloud-ext.yandex-team.ru"
QLOUD_DUMP_PATH = "/api/v1/environment/dump/"
QLOUD_UPDATE_PATH = "/api/v1/environment/upload/return-header?targetState=DEPLOYED&skipIfNoChange=false"

QLOUD_DUMP_URL = QLOUD_API_URL + QLOUD_DUMP_PATH
QLOUD_UPDATE_URL = QLOUD_API_URL + QLOUD_UPDATE_PATH

DEFAULT_VERSION_LIMIT = 10
SANDBOX_REOURCE_API = "https://sandbox.yandex-team.ru/api/v1.0/resource?"


def get_qloud_token(filename):
    with open(filename) as f:
        return f.read().strip()


def deploy_resource(args, environment=None):
    environment_name = args.environment_name
    if environment:
        environment_name = environment

    qloud_token = get_qloud_token(QLOUD_TOKEN_FILE)
    auth_headers = {"Authorization": "OAuth " + qloud_token}

    try:
        environment = requests.get(
            QLOUD_DUMP_URL + environment_name, headers=auth_headers).json()
    except ValueError as e:
        print "Can't find environment:  %s:\n%s" % (environment_name, str(e))
        return None

    environment_orig = copy.deepcopy(environment)

    environment["comment"] = "update monitorings: sandbox resource id %s" % args.resource_version
    for component in environment["components"]:
        component_name = component["componentName"]
        for resource in component["sandboxResources"]:
            if re.search(args.resource_name, resource["localName"]):
                resource["id"] = args.resource_version
                print "resource %s will be updated for %s " % (args.resource_name, component_name)
    diffs = deepdiff.DeepDiff(environment_orig, environment)

    if diffs.get("values_changed"):
        for item, changes in diffs.get("values_changed").iteritems():
            print "%s\n%s\n\n" % (item, changes)

    if args.dry_run is False:
        headers = auth_headers
        headers["content-type"] = "application/json"
        request_ans = requests.post(
            QLOUD_UPDATE_URL, headers=auth_headers, data=json.dumps(environment))
        if request_ans.status_code == 200:
            pprint(request_ans.json())


def utc_to_local(utc_dt):
    timestamp = calendar.timegm(utc_dt.timetuple())
    local_dt = datetime.fromtimestamp(timestamp)
    assert utc_dt.resolution >= timedelta(microseconds=1)
    return local_dt.replace(microsecond=utc_dt.microsecond)


def get_resource_info(resource_name, version_limit):
    post_data = {"limit": version_limit,
                 "offset": 0,
                 "attrs": json.dumps({"resource_name": resource_name})}
    resource_info = requests.get(
        SANDBOX_REOURCE_API + urlencode(post_data)).json()["items"]
    return resource_info


def print_resource(resource_info):
    for resource in resource_info:
        created_time = utc_to_local(datetime.strptime(
            resource["time"]["created"], '%Y-%m-%dT%H:%M:%SZ'))
        print "%d\t%s\t%s" % (resource["id"], created_time, resource["file_name"],)


def view_resource(args):
    resource_info = get_resource_info(args.resource_name, args.limit)
    print_resource(resource_info)


def load_environments(filename):
    with open(filename) as f:
        return f.read().splitlines()

def deploy(args):
    if not args.environment_filename:
        deploy_resource(args)
    else:
        environments = load_environments(args.environment_filename)
        for environment in environments:
            deploy_resource(args, environment=environment)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=("Sandbox and Qloud resource manager"))
    subparsers = parser.add_subparsers()
    view_parser = subparsers.add_parser("view")
    view_parser.add_argument("-n", "--resource-name",
                             type=str, required=True, help="Resource name")
    view_parser.add_argument("-l", "--limit", type=int, default=DEFAULT_VERSION_LIMIT,
                             required=False, help="API result limit")
    view_parser.set_defaults(func=view_resource)

    deploy_parser = subparsers.add_parser("deploy")
    env_group = deploy_parser.add_mutually_exclusive_group(required=True)
    env_group.add_argument("-e", "--environment-name", type=str,
                               help="Qloud environment name. f.e.: mail.mdbsave.production")
    env_group.add_argument("-E", "--environment-filename", type=str,
                               help="File with list of qloud environment names")
    deploy_parser.add_argument("-r", "--resource-version", type=int, default=0,
                               required=True, help="Resource version")
    deploy_parser.add_argument(
        "-n", "--resource-name", type=str, required=True, help="Resource name")
    deploy_parser.add_argument("-d", "--dry_run", action='store_true', default=False,
                               required=False, help="Dry run mode")
    deploy_parser.set_defaults(func=deploy)

    args = parser.parse_args()
    args.func(args)
