from pathlib import Path
from logging.handlers import RotatingFileHandler
from http.server import HTTPServer, SimpleHTTPRequestHandler
import logging
import socket
import json
import time
import os

UNISTAT_DIR = "/tmp/unistat/"
LOG_PATH = "/var/log/unistat/unistat.log"
PORT = 11010

log_size = 1024 * 1024 * 4


def setup_logging():
    log_path = Path(LOG_PATH)
    if not log_path.parent.exists():
        log_path.parent.mkdir()
    log_path.touch(exist_ok=True)
    loghandler = RotatingFileHandler(str(log_path), maxBytes=log_size, backupCount=1)
    LOGFORMAT = "timestamp=%(asctime)s\t%(message)s"

    logging.basicConfig(format=LOGFORMAT,
                        level=logging.INFO,
                        handlers=(loghandler,))
    logging.getLogger("urllib3").setLevel(logging.WARNING)


class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6


class UnistatHandler(SimpleHTTPRequestHandler):
    def log_message(self, format, *args):
        logging.info("ip=%s\tmethod=%s\trequest=%s\tcode=%s\texectime=%s",
                     self.client_address[0],
                     self.command,
                     self.path,
                     self.code,
                     "%.6f" % (time.time() - self.start_exec_time))

    def log_error(self, format, *args):
        if hasattr(self, "error_msg"):
            logging.info("ip=%s\trequest=%s\terror=%s",
                         self.client_address[0],
                         self.path,
                         self.error_msg)

    def send_response(self, code, message=None):
        self.code = code
        SimpleHTTPRequestHandler.send_response(self, code, message)

    def do_GET(self):
        self.start_exec_time = time.time()
        if self.path == "/unistat":
            try:
                response_body = self.unistat_response_body()
                self.send_response(200)
                self.send_header("Content-type", "text/json")
                self.end_headers()
                self.wfile.write(bytes(response_body, 'UTF-8'))
            except BaseException:
                self.send_response(500)
                self.end_headers()
                self.wfile.write(b"Failed to get unistat metrics")
        elif self.path == "/ping":
            self.send_response(200)
            self.end_headers()
            self.wfile.write(b"pong")
        else:
            self.send_response(404)
            self.end_headers()
            self.wfile.write(b"404 Not Found")

    def unistat_response_body(self):
        unists_files = [os.path.join(UNISTAT_DIR, f) for f in os.listdir(UNISTAT_DIR) if os.path.isfile(os.path.join(UNISTAT_DIR, f))]
        unistat_data = []
        for file in unists_files:
            with open(file) as f:
                unistat_data += json.load(f)
        return json.dumps(unistat_data)


def main():
    setup_logging()
    server = HTTPServerV6(('::', PORT), UnistatHandler)
    server.serve_forever()


if __name__ == '__main__':
    main()
