import argparse
import datetime
import re
import yaml
import sqlite3
from yql.api.v1.client import YqlClient

yesterday = str(datetime.date.today() - datetime.timedelta(days=1))
two_month_ago = str(datetime.date.today() - datetime.timedelta(days=60))


def load_config(cfg_path):
    with open(cfg_path) as f:
        return yaml.safe_load(f)


def check_db(config):
    check_db_query = "SELECT MAX(date) FROM stat;"
    conn = sqlite3.connect(config["db_path"])
    c = conn.cursor()

    # Проверяем наличие таблицы stat:
    try:
        c.execute(check_db_query)
    except sqlite3.OperationalError:
        return False
    db_date = c.fetchall()[0][0]
    conn.close()

    # Проверяем наличие записей в stat:
    if not db_date:
        return False

    return db_date.split()[0]


def create_schema(config):
    conn = sqlite3.connect(config["db_path"])
    c = conn.cursor()
    c.execute(
        """CREATE TABLE IF NOT EXISTS stat(
        [abc_service] TEXT,
        [host] TEXT,
        [service] TEXT,
        [login] TEXT,
        [method] TEXT,
        [date] DATE,
        [unixtime] DATETIME,
        [delay] INTEGER);""")
    conn.commit()
    conn.close()


def update_db_from_yql(config, date_from, date_to):
    abc_services_quoted = ", ".join(
        '"' + i + '"' for i in config["abc_services"])
    client = YqlClient()
    yql_query = """
               USE hahn;
               PRAGMA yt.InferSchema;
               SELECT abc_service, checks, login, method, iso_eventtime, unixtime, sent_delay
               FROM RANGE(`//home/logfeller/logs/juggler-banshee-log/1d`, "DATE_FROM","DATE_TO")
               WHERE abc_service IN (ABC_SERVICES) AND method IN ("phone_escalation", "telegram");"""
    yql_query = re.sub("ABC_SERVICES", abc_services_quoted, yql_query)
    yql_query = re.sub("DATE_FROM", date_from, yql_query)
    yql_query = re.sub("DATE_TO", date_to, yql_query)
    print yql_query

    request = client.query(yql_query, syntax_version=1)
    request.run()
    result = request.get_results()

    insert_query = """
        INSERT INTO stat VALUES("abc_service",
        "host",
        "service",
        "login",
        "method",
        "date",
        unixtime,
        sent_delay);"""

    conn = sqlite3.connect(config["db_path"])
    c = conn.cursor()
    for table in result:
        table.fetch_full_data()
        for row in table.rows:
            abc_service, checks, login, method, date, unixtime, sent_delay = row
            sent_delay = re.sub("s", "", sent_delay)
            host = checks.split(":")[0]
            service = checks.split(":")[1]
            query = re.sub("abc_service", abc_service, insert_query)
            query = re.sub("host", host, query)
            query = re.sub("service", service, query)
            query = re.sub("login", login, query)
            query = re.sub("method", method, query)
            query = re.sub("date", date, query)
            query = re.sub("unixtime", str(int(float(unixtime))), query)
            query = re.sub("sent_delay", str(int(float(sent_delay))), query)
            c.execute(query)
    conn.commit()
    conn.close()


def main(args, config):
    if check_db(config):
        db_max_date = check_db(config)
        if db_max_date < yesterday:
            db_max_date = datetime.datetime.strptime(db_max_date, '%Y-%m-%d').date()
            day_after_db_max_date = str(db_max_date + datetime.timedelta(days=1))
            update_db_from_yql(config, day_after_db_max_date, yesterday)
        else:
            print "Data in local db is already actual"
    else:
        create_schema(config)
        update_db_from_yql(config, two_month_ago, yesterday)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description=("Fetch esacalations from YT"))
    parser.add_argument("-c", "--config", type=str,
                        help="Path to config", required=True)
    args = parser.parse_args()
    config = load_config(args.config)
    main(args, config)
