#!/usr/bin/python
# -*- coding: UTF-8 -*-

import argparse
import bottle
import datetime
import re
import sqlite3
import yaml
from bottle import route, run, request, redirect, template


TEMPLATES_DIR = "/usr/share/escalations/"
ESCALATION_ALERT_TIME = 300
ESCALATION_SHOW_TIME = 60
DAY_HOURS = range(8, 22)


ALL_ESCALATIONS_QUERY = """
    SELECT
        host, service, login, method, date
    FROM stat 
    WHERE 
        date > "DATE_FROM" AND
        date < "DATE_TO" AND
        abc_service = "ABC_SERVICE" AND
        method in (METHODS)
    ORDER BY date;"""

TOP_ESCALATIONS_QUERY = """
    SELECT
        host, service, COUNT(*) AS count
    FROM stat
    WHERE
        date > "DATE_FROM" AND
        date < "DATE_TO" AND
        abc_service = "ABC_SERVICE" AND
        method in (METHODS)
    GROUP BY host, service
    ORDER BY count DESC;"""

types_queries = {"all": ALL_ESCALATIONS_QUERY, "top": TOP_ESCALATIONS_QUERY}


def load_config(cfg_path):
    with open(cfg_path) as f:
        return yaml.safe_load(f)


def prepare_query(query, daterange, abc_service, methods):
    methods = [method for method, state in methods.iteritems()
               if state]
    methods = ", ".join('"' + i + '"' for i in methods)
    query = re.sub("DATE_FROM", daterange["date_from"], query)
    query = re.sub("DATE_TO", daterange["date_to"], query)
    query = re.sub("ABC_SERVICE", abc_service, query)
    query = re.sub("METHODS", methods, query)
    return query


def load_escalations_from_db(sort_type, daterange, abc_service, methods):

    query = prepare_query(
        types_queries[sort_type], daterange, abc_service, methods)
    print query

    conn = sqlite3.connect(config["db_path"])
    c = conn.cursor()
    c.execute(query)
    data = c.fetchall()
    conn.close()

    escalations = []
    if sort_type == "all":
        for row in data:
            host, service, login, method, date = row
            escalations.append({"host": host, "service": service,
                                "login": login, "method": method, "date": date})
    elif sort_type == "top":
        for row in data:
            host, service, count = row
            escalations.append({"host": host, "service": service,
                                "count": count})
    return escalations


@route('/', method='GET')
def root():
    abc_service = request.GET.get("abc_service")
    return template("stat", route="main", abc_service=abc_service,daterange_link="", methods={}, sort_type=None, abc_services=config["abc_services"])


@route('/stat', method='GET')
def stat():
    today = datetime.date.today()
    days_from_monday = datetime.datetime.today().isoweekday()
    monday = str(today - datetime.timedelta(days=days_from_monday - 1))
    today = str(today)
    abc_service = request.GET.get("abc_service")
    sort_type = request.GET.get("sort", "all")

    phone_escalation = request.GET.get("phone_escalation")
    telegram = request.GET.get("telegram")
    if not phone_escalation and not telegram:
        phone_escalation = 1
        telegram = 1
    methods = {"phone_escalation": phone_escalation, "telegram": telegram}

    daterange = request.GET.get("daterange")

    if daterange:
        daterange = date_from, date_to = daterange.split(" 一 ")
        daterange = {"date_from": date_from, "date_to": date_to}
    else:
        daterange = {"date_from": monday, "date_to": today}
        if monday > today:
            daterange["date_to"] = monday

    escalations = load_escalations_from_db(
        sort_type, daterange, abc_service, methods)
    daterange_link = daterange["date_from"] + "+一+" + daterange["date_to"]
    return template("stat", methods=methods, abc_service=abc_service, escalations=escalations,
                    daterange=daterange, route="stat", sort_type=sort_type, abc_services=config["abc_services"],
                    daterange_link=daterange_link)


if __name__ == "__main__":
    bottle.TEMPLATE_PATH.insert(0, TEMPLATES_DIR)
    parser = argparse.ArgumentParser(description=("Escalations web UI"))
    parser.add_argument("-c", "--config", type=str,
                        help="Path to config", required=True)
    args = parser.parse_args()

    config = load_config(args.config)
    run(host='::', port=8080, debug=True)
