import os
from enum import Enum

import yaml

from library.python import resource
from mail.devpack.ctl.lib.main import wrap_main
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import DogAppComponent, FakeRootComponent
from mail.webmail_config.lib.make_config import make_config
from mail.devpack.lib.components.fakebb import FakeBlackbox
from mail.pg.mopsdb.devpack.components.mopsdb import MopsDb
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvmapi import TvmApi
from mail.devpack.lib.components.unimock import Unimock


class MarkStatus(Enum):
    read = 0
    not_read = 1
    replied = 2
    forwarded = 3


ALIASES = {'from_': 'from', 'type_': 'type'}


def make_args_dict(**kwargs):
    def dealias(key, value):
        return ALIASES.get(key, key), str(value)

    return dict(dealias(k, v) for k, v in kwargs.items() if v is not None)


def flatten(array):
    return ','.join(str(value) for value in array) if array is not None else None


class MopsApi(object):
    def __init__(self, yhttp, service_ticket, **default_args):
        self.yhttp = yhttp
        self.service_ticket = service_ticket
        self.default_args = default_args

    def _fill_by_default(self, **kwargs):
        for k, v in self.default_args.items():
            if k not in kwargs:
                kwargs[k] = v

        return kwargs

    def _make_operation_data(self, uid=None, ip=None, login=None, karma=None, karma_status=None,
                             source=None, country=None, ora_ip=None, ora_connection_id=None, session_info=None,
                             yandexuid=None, icookie=None, ua=None, mids=None, tids=None, fid=None, subject=None,
                             from_=None, age=None):
        return make_args_dict(uid=uid, ip=ip, login=login, karma=karma, karma_status=karma_status,
                              source=source, country=country, ora_ip=ora_ip, ora_connection_id=ora_connection_id,
                              session_info=session_info, yandexuid=yandexuid, icookie=icookie, ua=ua,
                              mids=flatten(mids), tids=flatten(tids), fid=fid, subject=subject, from_=from_, age=age)

    def _operation(self, target, args=None, **common_args):
        defaulted_common_args = self._fill_by_default(**common_args)
        data = self._make_operation_data(**defaulted_common_args)
        data.update(args)
        return self.yhttp.post(target=target, data=data, headers={'X-Ya-Service-Ticket': self.service_ticket})

    def _with_sent_aware_operation(self, target, args=None, with_sent=None, **common_args):
        if with_sent:
            if args:
                args['with_sent'] = with_sent
            else:
                args = {'with_sent': with_sent}
        return self._operation(target=target, args=args, **common_args)

    def _make_basic_operation_data(self, uid=None):
        return make_args_dict(uid=uid)

    def _basic_operation(self, target, args, **common_args):
        defaulted_common_args = self._fill_by_default(**common_args)
        data = self._make_basic_operation_data(**defaulted_common_args)
        data.update(args)
        return self.yhttp.post(target=target, data=data, headers={'X-Ya-Service-Ticket': self.service_ticket})

    def ping(self):
        return self.yhttp.ping()

    def stat(self, uid, **common_args):
        return self.yhttp.get('/stat', params={'uid': uid}, **common_args)

    def spam(self, nomove=None, **common_args):
        args = make_args_dict(nomove=nomove)
        return self._with_sent_aware_operation(target='spam', args=args, **common_args)

    def unspam(self, dest_fid=None, nomove=None, **common_args):
        args = make_args_dict(dest_fid=dest_fid, nomove=nomove)
        return self._operation(target='/unspam', args=args, **common_args)

    def purge(self, **common_args):
        return self._operation(target='/purge', args={}, **common_args)

    def purge_hidden_trash(self, **common_args):
        return self._operation(target='/purge_hidden_trash', args={}, **common_args)

    def remove(self, nopurge=None, **common_args):
        args = make_args_dict(nopurge=nopurge)
        return self._with_sent_aware_operation(target='/remove', args=args, **common_args)

    def complex_move(self, dest_fid, **common_args):
        args = make_args_dict(dest_fid=dest_fid)
        return self._with_sent_aware_operation(target='/complex_move', args=args, **common_args)

    def mark(self, status, postmaster=None, **common_args):
        args = make_args_dict(status=status.name, postmaster=postmaster)
        return self._operation(target='/mark', args=args, **common_args)

    def label(self, lids, **common_args):
        return self._operation(target='/label', args={'lids': flatten(lids)}, **common_args)

    def unlabel(self, lids, **common_args):
        return self._operation(target='/unlabel', args={'lids': flatten(lids)}, **common_args)

    def create_folder(self, name, parent_fid=None, symbol=None, strict=None, **common_args):
        args = make_args_dict(name=name, parent_fid=parent_fid, symbol=symbol, strict=strict)
        return self._basic_operation(target='/folders/create', args=args, **common_args)

    def create_hidden_trash_folder(self, **common_args):
        return self._basic_operation(target='/folders/create_hidden_trash', args={}, **common_args)

    def update_folder(self, fid, name=None, parent_fid=None, **common_args):
        args = make_args_dict(fid=fid, name=name, parent_fid=parent_fid)
        return self._basic_operation(target='/folders/update', args=args, **common_args)

    def update_folder_symbol(self, fid, symbol, **common_args):
        args = {'fid': fid, 'symbol': symbol}
        return self._basic_operation(target='/folders/update_symbol', args=args, **common_args)

    def update_folder_position(self, fid, prev_fid, **common_args):
        args = {'fid': fid, 'prev_fid': prev_fid}
        return self._basic_operation(target='/folders/update_position', args=args, **common_args)

    def delete_folder(self, fid, **common_args):
        return self._basic_operation(target='/folders/delete', args={'fid': fid}, **common_args)

    def update_pop3_folder(self, fids, **common_args):
        return self._basic_operation(target='/folders/update_pop3', args={'fids': flatten(fids)}, **common_args)

    def create_label(self, symbol=None, name=None, color=None, type_=None, **common_args):
        args = make_args_dict(symbol=symbol, name=name, color=color, type_=type_)
        return self._basic_operation(target='/labels/create', args=args, **common_args)

    def update_label(self, lid, name=None, color=None, **common_args):
        args = make_args_dict(lid=lid, name=name, color=color)
        return self._basic_operation(target='/labels/update', args=args, **common_args)

    def delete_label(self, lid, **common_args):
        return self._basic_operation(target='/labels/delete', args={'lid': lid}, **common_args)


class Mops(DogAppComponent):
    NAME = "mops"
    DEPS = [Sharpei, MopsDb, Unimock, TvmApi, FakeBlackbox]

    def __init__(self, env, components):
        super(Mops, self).__init__(env, components, custom_path='mops')

    def init_root(self):
        self.yhttp.init_root()

        helpers.write2file(resource.find('macs_pg/query.conf'), os.path.join(self.config_path, 'query.conf'))
        helpers.write2file(resource.find('mops/pq_query.conf'), os.path.join(self.config_path, 'pq_query.conf'))
        helpers.write2file(resource.find('mops/tvm_secret'), os.path.join(self.secrets_path, 'tvm_secret'))
        helpers.write2file(resource.find('mops/stoken'), os.path.join(self.secrets_path, 'stoken'))

        self._init_root()

    def _init_root(self):
        self._generate_config('development')

        replace = {
            'blackbox_port': self.components[FakeBlackbox].port,
            'sharpei_port': self.components[Sharpei].webserver_port(),
            'mopsdb_port': self.components[MopsDb].port(),
            'postmaster_port': self.components[Unimock].port,
            'pusher_port': self.components[Unimock].port,
            'tvm_port': self.components[TvmApi].port
        }

        devpack = self.yhttp.format_config(resource.find('mops/devpack.yml'), **replace)
        helpers.write2file(devpack, os.path.join(self.config_path, 'config-devpack.yml'))

        with open('mops.properties-devpack', 'w') as f:
            f.write('''
mops.host=http://localhost
mops.port={mops_port}
                '''.format(mops_port=self.webserver_port()))

    def _generate_config(self, env):
        base = resource.find('mops/base.yml')
        service_yaml = yaml.safe_load(resource.find('mops/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        config = self.yhttp.format_config(make_config(env, base, service_yaml, common_yaml, silent=True))
        helpers.write2file(config, os.path.join(self.config_path, 'config.yml'))

    def start(self, put_pgpass_in_env=True):
        super(Mops, self).start(put_pgpass_in_env=put_pgpass_in_env)

    def api(self, **default_args):
        return MopsApi(self.yhttp, self.components[TvmApi].get_ticket('tests', 'mops'), **default_args)


class MopsTesting(Mops):
    NAME = 'mops_testing'
    DEPS = [MopsDb]

    def _init_root(self):
        config = resource.find('mops/testing.yml')
        config = self.yhttp.format_config(
            config,
            mopsdb_port=self.components[MopsDb].port())
        helpers.write2file(config, os.path.join(self.config_path, 'config-devpack.yml'))

        self._generate_config('testing')
        self.yhttp.init_pgpass(self.get_root())

        with open('mops.properties-testing', 'w') as f:
            f.write('''
mops.host=http://localhost
mops.port={mops_port}
                '''.format(mops_port=self.webserver_port()))

    def start(self):
        super(MopsTesting, self).start(put_pgpass_in_env=True)


class MopsService(FakeRootComponent):
    NAME = 'mops-service'
    DEPS = [Mops, MopsTesting]


def main():
    wrap_main(deps_root=MopsService)
