#pragma once

#include <mailbox_oper/mailbox_oper.h>
#include <ymod_taskmaster/service.hpp>
#include <ymod_taskmaster/marshalling.hpp>
#include <ymod_taskmaster/errors.hpp>
#include <yplatform/find.h>
#include <internal/logger.h>
#include <internal/server/task_traits.h>

namespace mbox_oper {

using Result = ymod_taskmaster::Service::Result;

class MailboxOperAsyncApi {
    using TaskTraitsOpt = boost::optional<TaskTraits>;
    using TaskPtr = ymod_taskmaster::TaskPtr;
    using SourcePtr = std::shared_ptr<mbox_oper::MidsSource>;
public:
    MailboxOperAsyncApi(MailboxOper& moperApi, const MailboxOperParams& commonParams, ContextPtr context)
        : moperApi_(moperApi), commonParams_(commonParams), logger_(getContextLogger(context)),
          context_(std::move(context)), taskMaster_( yplatform::find<ymod_taskmaster::Service, std::shared_ptr>("task_master") )
    {}

    template <typename OpTask>
    TaskTraits execute(typename OpTask::ParamsType params, SourcePtr source, YieldCtx yield) {
        constexpr auto type = OpTask::typeValue;
        const auto resolveOptions = params.resolveOptions(moperApi_.meta(), commonParams_, yield);
        const auto action = ymod_taskmaster::toString(type);
        const bool breakOnEmptyMids = params.breakOnEmptyMids();

        const auto result = tryPlanTask<OpTask>(params, source, yield);
        if (result) {
            return result.get();
        }

        const auto mids = source->resolve(moperApi_.meta(), resolveOptions, yield);
        logMids(action, mids, type);
        const auto convertedMids = ymod_taskmaster::convertMids(mids);

        if (mids.empty() && breakOnEmptyMids) {
            LOGDOG_(logger_, notice, log::message="mids list empty, will do nothing", log::action=action);
            return TaskTraits::sync();
        }

        auto task = std::make_unique<OpTask>(commonParams_, std::move(params));

        const auto logBackoff = [&](const char* reason, const std::exception& e) {
            LOGDOG_(logger_, notice, log::message="will put task to queue",
                    log::reason=reason, log::action=action, log::exception=e);
        };

        try {
            logSyncAction(action);
            task->execute(moperApi_, convertedMids, false, context_, yield);
            return TaskTraits::sync();
        } catch (const ReadOnlyMetabaseException& e) {
            logBackoff("DB is RO", e);
        } catch (const pgg::system_error& e) {
            if (e.code() == pgg::errc::databaseLockFailed) {
                logBackoff("DB lock timeout", e);
            } else if (e.code() == pgg::errc::noEndpointFound) {
                logBackoff("DB endpoints unavailable", e);
            } else {
                throw;
            }
        }

        const auto taskGroupId = taskMaster_->planTask(std::move(task), convertedMids,
                context_, yield);
        logAsyncAction(action, taskGroupId);
        return TaskTraits::async(taskGroupId);
    }
private:
    void logMids(const std::string& action, const Mids& mids, const ymod_taskmaster::TaskType oper) {
        LOGDOG_(logger_, notice, log::action=action, log::uid=commonParams_.uid,
                log::operation=oper, log::mids=mids)
    }

    void logSyncAction(const std::string& action) {
        LOGDOG_(logger_, notice, log::message="will act synchronously", log::action=action);
    }

    void logAsyncAction(const std::string& action, const ymod_taskmaster::TaskGroupId taskGroupId) {
        LOGDOG_(logger_, notice, log::message="put task in queue",
                log::action=action, log::task_group_id=taskGroupId);
    }

    template <typename OpTask>
    TaskTraitsOpt tryPlanTask(const typename OpTask::ParamsType& params, SourcePtr source, YieldCtx yield) {
        try {
            constexpr auto type = OpTask::typeValue;
            const auto action = ymod_taskmaster::toString(type);
            const auto resolveOptions = params.resolveOptions(moperApi_.meta(), commonParams_, yield);

            Result res;
            if (source->isNeedAsyncResolve(resolveOptions)) {
                using ymod_taskmaster::ResolveMidsTask;

                const size_t upperBound = source->getUpperMidsCount(moperApi_.meta(), resolveOptions, yield);
                if (source->isNeedPaginate()) {
                    auto taskMaster = yplatform::find<ymod_taskmaster::Service>("task_master");
                    source = source->paginate(std::nullopt, taskMaster->getQueueSettings().max_mids_to_resolve);
                }
                LOGDOG_(logger_, notice, log::message="will try to put task in queue with resolve",
                        log::action=action, log::upper_bound=upperBound);
                auto resolveTask = std::make_shared<ResolveMidsTask>(commonParams_, std::move(source), params);
                res = taskMaster_->planTaskIfAsync(std::move(resolveTask), upperBound, context_, yield);
            } else {
                const auto mids = source->resolve(moperApi_.meta(), resolveOptions, yield);
                const auto convertedMids = ymod_taskmaster::convertMids(mids);
                auto task = std::make_shared<OpTask>(commonParams_, params);
                res = taskMaster_->planTaskIfAsync(std::move(task), convertedMids, context_, yield);

                if (res) {
                    logMids(action, mids, type);
                }
            }

            if (res) {
                logAsyncAction(action, *res);
                return TaskTraits::async(*res);
            }
        } catch (const ymod_taskmaster::LimitException&) {
            throw;
        } catch (const std::exception& e) {
            LOGDOG_(logger_, notice, log::message="error while planning task", log::exception=e);
        }

        return TaskTraitsOpt();
    }

    MailboxOper& moperApi_;
    const MailboxOperParams commonParams_;
    ContextLogger logger_;
    ContextPtr context_;
    const ymod_taskmaster::ServicePtr taskMaster_;
};

}
